/*
 * File: log.c
 *
 * General logging and tracing routines
 *
 * Bob Eager   May 2016
 *
 */

#include <stdarg.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <time.h>
#include <sys/types.h>

#include <errno.h>
#include <syslog.h>

#include "log.h"

#ifdef	DEBUG
#define	MAXTRACE	500		/* Maximum length of trace line */
#endif

#define	SYSLOGSERVICE	"syslog"	/* Name of syslog service */
#define	UDP		"udp"		/* UDP protocol */

/* Forward references */

static	VOID	dolog_file(UINT, PCHAR);
static	VOID	dolog_syslog(UINT, PCHAR);
static	INT	open_logfile(PCHAR);
static	INT	open_syslog(PCHAR, PCHAR);

/* Local storage */

static	FILE	*logfp = (FILE *) NULL;
static	UINT	logging_type = LOGGING_UNSET;


/*
 * Open the logging system. The 'type' parameter specifies how the logging
 * is to be done - to a file, or to the syslog daemon on the local machine.
 * 'file' specifies the logfile.
 *
 * Returns:
 *	LOGERR_OK		log successfully opened
 *	LOGERR_OPENFAIL		failed to open log
 *
 */

INT open_log(UINT log_type, PCHAR file, PCHAR myname,
		PCHAR myprocname)
{	logging_type = log_type;

	switch(log_type) {
		case LOGGING_FILE:
			return(open_logfile(file));

		case LOGGING_SYSLOG:
			return(open_syslog(myname, myprocname));

		default:
			logging_type = LOGGING_UNSET;
			return(LOGERR_LOGTYPE);
	}
}


/*
 * Open the logfile.
 *
 * 'file' specifies the logfile.
 *
 *
 * Returns:
 *	LOGERR_OK		log successfully opened
 *	LOGERR_OPENFAIL		failed to open log
 *
 */

static INT open_logfile(PCHAR file)
{	logfp = fopen(file, "a");
	if(logfp == (FILE *) NULL) {
		fprintf(stderr, "logfile failure: %d\n", errno);
		return(LOGERR_OPENFAIL);
	}

	return(LOGERR_OK);
}


/*
 * Open the syslog.
 *
 * Returns:
 *	LOGERR_OK		log successfully opened
 *	LOGERR_NOENV		environment variable not set for log directory
 *	LOGERR_OPENFAIL		failed to open log
 *
 */

static INT open_syslog(PCHAR myname, PCHAR myprocname)
{	openlog(myprocname, LOG_PID, LOG_MAIL);
	return(LOGERR_OK);
}


/*
 * Close the log.
 *
 */

VOID close_log(VOID)
{	switch(logging_type) {
		case LOGGING_FILE:
			fclose(logfp);
			logfp = (FILE *) NULL;
			break;

		case LOGGING_SYSLOG:
			closelog();
			break;
	}
	logging_type = LOGGING_UNSET;
}


/*
 * Write a string to the log, wherever it is.
 *
 */

VOID dolog(UINT type, PCHAR s)
{	switch(logging_type) {
		case LOGGING_FILE:
			dolog_file(type, s);
			break;

		case LOGGING_SYSLOG:
			dolog_syslog(type, s);
			break;
	}
}


/*
 * Write a string to the logfile. The string is timestamped, and a newline
 * appended to the end unless there is one there already.
 *
 * This routine is thread-safe; it writes only ONE string to the logfile,
 * ensuring that the file does not become garbled.
 *
 */

static VOID dolog_file(UINT type, PCHAR s)
{	time_t tod;
	CHAR timeinfo[35];
	CHAR buf[MAXLOG+1];

	if(logfp == (FILE *) NULL) return;

	(VOID) time(&tod);
	(VOID) strftime(
			timeinfo,
			sizeof(timeinfo),
			"%d/%m/%y %X>",
			localtime(&tod));
	sprintf(buf, "%s %s", timeinfo, s);
	if(s[strlen(s)-1] != '\n') strcat(buf, "\n");

	fputs(buf, logfp);
	fflush(logfp);
}


#pragma	clang diagnostic push
#pragma	clang diagnostic ignored "-Wformat-security"
static VOID dolog_syslog(UINT severity, PCHAR s)
{	syslog(severity | LOG_MAIL, s);
}
#pragma	clang diagnostic pop

#ifdef	DEBUG
/*
 * Output trace message, in printf style, to the logfile.
 *
 */

VOID trace(PCHAR mes, ...)
{	va_list ap;
	CHAR buf[MAXTRACE+1];

	strcpy(buf, "trace: ");

	va_start(ap, mes);
	vsprintf(buf+strlen(buf), mes, ap);
	va_end(ap);

	dolog(LOG_DEBUG, buf);
}
#endif

/*
 * End of file: log.c
 *
 */
