/*
 * File: md5.c
 *
 * MD5 implementation
 *
 * Bob Eager   May 2016
 *
 */

#include <stdlib.h>
#include <string.h>
#include <sys/types.h>

#include "typedefs.h"
#include "md5.h"

#define	IPAD		0x36
#define	OPAD		0x5c
#define	SIZE		64

/*
 * Generate a digest of 'text' using 'key' as a key, returning the result
 * in 'digest'.
 *
 */

VOID hmac_md5(PCHAR key, INT key_len, PCHAR text, INT text_len, PCHAR digest)
{	INT i;
	MD5_CTX context;
	CHAR k_ipad[SIZE+1];		/* inner padding; key XORd with IPAD */
	CHAR k_opad[SIZE+1];		/* outer padding; key XORd with OPAD */
	CHAR tk[DIGEST_SIZE];		/* temporary key digest */

	/* If key is longer than 64 bytes reset it to key=MD5(key) */

	if(key_len > SIZE) {
		MD5_CTX tctx;		/* temporary context */

		MD5Init(&tctx);
		MD5Update(&tctx, key, key_len);
		MD5Final(tk, &tctx);

		key = tk;		/* use digest as key now */
		key_len = DIGEST_SIZE;
	}

	/*
	 * The HMAC_MD5 transform looks like:
	 *
	 * MD5(K XOR opad, MD5(K XOR ipad, text))
	 *
	 * where K is an n byte key
	 * ipad is the byte 0x36 repeated 64 times
	 * opad is the byte 0x5c repeated 64 times
	 * and text is the data being protected
	 *
	 */

	/* Start out by storing key in pads */

	memset(k_ipad, '\0', sizeof(k_ipad));
	memset(k_opad, '\0', sizeof(k_opad));
	memcpy(k_ipad, key, key_len);
	memcpy(k_opad, key, key_len);

	/* XOR key with IPAD and OPAD values */

	for(i = 0; i < SIZE; i++) {
		k_ipad[i] ^= IPAD;
		k_opad[i] ^= OPAD;
	}

	/* Perform inner MD5 */

	MD5Init(&context);		/* Init context for 1st pass */
	MD5Update(&context, k_ipad, SIZE);
					/* Start with inner pad */
	MD5Update(&context, text, text_len);
					/* then text of datagram */
	MD5Final(digest, &context);	/* finish up 1st pass */

	/* Perform outer MD5 */

	MD5Init(&context);		/* Init context for 2nd pass */
	MD5Update(&context, k_opad, SIZE);
					/* Start with outer pad */
	MD5Update(&context, digest, DIGEST_SIZE);
					/* then results of 1st hash */
	MD5Final(digest, &context);	/* finish up 2nd pass */
}

/*
 * End of file: md5.c
 *
 */

