/*
 * File: smtp.c
 *
 * Bare SMTP client
 *
 * Main program
 *
 * Bob Eager   May 2016
 *
 */

/*
 * History:
 *
 *	1.0	Initial version.
 *	1.1	Use new thread-safe logging module.
 *		Use OS/2 type definitions.
 *		Added -v option to display progress of mail transmission.
 *	1.2	New, simplified network interface module.
 *	1.3	Does not make connection at all if there are no messages
 *		to send.
 *	1.4	Corrected failure to dot-stuff on sending.
 *	1.5	Modified to use NETLIB DLL.
 *	2.0	Added BLDLEVEL string.
 *		Diagnostics for occasional logfile open failures.
 *		Additional error checking in logging module.
 *		Grouped initialisation code together.
 *	3.0	Recompiled using 32-bit TCP/IP toolkit, in 16-bit mode.
 *	3.1	Added support for ESMTP function AUTH (LOGIN, PLAIN).
 *	4.0	Added support for logging to 'syslog' instead of to file,
 *		selectable by '-z' option. This has the advantage of
 *		avoiding clashing logfile usage.
 *	4.1	Fixed error in handling an EHLO response where there are
 *		no SMTP extensions reported.
 *	4.2	Added -q flag for quiet operation.
 *	4.3	Corrected exit code inversion (0 for failure!).
 *	4.4	Added support for sending ETRN to server.
 *	4.5	Fixed problem where multiline replies to the initial connect
 *		were not being handled properly.
 *		Fixed problem when an unsupported authorisation method could be
 *		mistaken as supported.
 *	4.6	Added CRAM-MD5 as an authentication mechanism.
 *	4.7	Gave LOGIN authentication priority over PLAIN if both are
 *		supported by the server (this was the 4.5 behaviour).
 *	5.0	Ported to FreeBSD.
 *		Allowed specification of alternate logfile.
 *		Allowed specification of destination port.
 *	5.1	Fixed select problem (wrong nfds) for FreeBSD 8.3.
 *	5.2	Minor fixes for clang compiler.
 *
 */

/*
 * For details of the protocol, see RFC 5321.
 *
 */

#include <ctype.h>
#include <stdlib.h>
#include <stdarg.h>
#include <stdio.h>
#include <string.h>
#include <time.h>
#include <libgen.h>
#include <limits.h>
#include <sys/types.h>

#include <sys/socket.h>
#include <netinet/in.h>
#include <netdb.h>
#include <arpa/nameser.h>
#include <arpa/inet.h>
#include <resolv.h>
#include <unistd.h>

#include "smtp.h"

#define	MAXLOGFILENAME	200		/* Max length of logfile name */
#define	MAXPORTSTR	100		/* Max length of port number string */
#define	LOGFILE		"baresmtp.log"	/* Name of log file */
#define	SMTPDIR		"BARESMTP"	/* Environment variable for spool dir */
#define	SMTPSERVICE	"smtp"		/* Name of SMTP service */
#define	TCP		"tcp"		/* TCP protocol */

/* Forward references */

static	VOID	add_directory(PCHAR);
static	VOID	add_file(PCHAR);
static	VOID	fix_domain(PCHAR);
static	VOID	log_connection(PCHAR, BOOL);
static	VOID	putusage(VOID);

/* Local storage */

static	LOGTYPE	log_type = LOGGING_UNSET;
static	PFL	head = (PFL) NULL;	/* Head of file list */
static	PFL	tail;
static	CHAR	progname[50];		/* Name of program, as a string */

/* Help text */

static	const	PCHAR helpinfo[] = {
"%s: SMTP client",
"Synopsis: %s [options] [file...]",
" Options:",
"    -ddirectory  specify directory containing mail; all files are sent",
"    -edomain     send ETRN for domain",
"    -h           display this help",
"    -l logname   specify logfile name (default is logging to syslog)",
"    -P port      specify alternate port for connection",
"    -ppass       specify password for authentication",
"    -q           operate quietly",
"    -sserver     specify address of SMTP server",
"    -uuser       specify username for authentication",
"    -v           verbose; display progress",
" ",
"Any specified file is treated as a single mail message.",
" ",
"If no files or directories are specified, the directory described",
"by the environment variable "SMTPDIR" is used.",
"There is no default for the address of the SMTP server.",
"Sending mail and sending ETRN are mutually exclusive.",
""
};


/*
 * Parse arguments and handle options.
 *
 */

INT main(INT argc, PCHAR argv[])
{	INT sockno, rc, port, i;
	BOOL verbose = FALSE, quiet = FALSE;
	PCHAR argp;
	CHAR logname[MAXLOGFILENAME+1];
	CHAR servername[MAXDNAME+1];
	CHAR clientname[MAXDNAME+1];
	CHAR username[MAXUNAME+1];
	CHAR password[MAXPASS+1];
	CHAR domain[MAXDNAME+1];
	CHAR temp_port[MAXPORTSTR+1];
	ULONG server_addr;
	SOCK server;
	PHOST smtphost;
	PSERV smtpserv;

	strcpy(progname, basename(argv[0]));

	tzset();			/* Set time zone */
	res_init();			/* Initialise resolver */

	logname[0] = '\0';
	servername[0] = '\0';
	username[0] = '\0';
	password[0] = '\0';
	domain[0] = '\0';
	temp_port[0] = '\0';

	/* Get default port number */

	smtpserv = getservbyname(SMTPSERVICE, TCP);
	if(smtpserv == (PSERV) NULL) {
		error("cannot get port for %s/%s service", SMTPSERVICE, TCP);
		exit(EXIT_FAILURE);
	}
	endservent();
	port = ntohs(smtpserv->s_port);

	/* Process input options */

	for(i = 1; i < argc; i++) {
		argp = argv[i];
		if(argp[0] == '-') {		/* Option */
			switch(argp[1]) {
				case 'd':	/* Specified directory */
					if(argp[2] != '\0') {
						add_directory(&argp[2]);
					} else {
						if(i == argc - 1) {
							error("no arg for -d");
							exit(EXIT_FAILURE);
						} else {
							add_directory(
								argv[++i]);
						}
					}
					break;

				case 'e':	/* Send ETRN for domain */
					if(domain[0] != '\0') {
						error(
							"ETRN domain specified"
							" more than once");
						exit(EXIT_FAILURE);
					}
					if(argp[2] != '\0') {
						strcpy(domain, &argp[2]);
					} else {
						if(i == argc - 1) {
							error("no arg for -e");
							exit(EXIT_FAILURE);
						} else {
							strcpy(
								domain,
								argv[++i]);
						}
					}
					break;

				case 'h':	/* Display help */
					putusage();
					exit(EXIT_SUCCESS);

				case 'l':	/* Specified logfile */
					if(logname[0] != '\0') {
						error(
							"logfile specified"
							" more than once");
						exit(EXIT_FAILURE);
					}
					if(argp[2] != '\0') {
						strcpy(logname, &argp[2]);
					} else {
						if(i == argc - 1) {
							error("no arg for -l");
							exit(EXIT_FAILURE);
						} else {
							strcpy(
								logname,
								argv[++i]);
						}
					}
					break;

				case 'P':	/* Specified port */
					if(temp_port[0] != '\0') {
						error(
							"port specified"
							" more than once");
						exit(EXIT_FAILURE);
					}
					if(argp[2] != '\0') {
						strcpy(temp_port, &argp[2]);
					} else {
						if(i == argc - 1) {
							error("no arg for -P");
							exit(EXIT_FAILURE);
						} else {
							strcpy(
								temp_port,
								argv[++i]);
						}
					}

					break;

				case 'p':	/* Specified password */
					if(password[0] != '\0') {
						error(
							"password specified"
							" more than once");
						exit(EXIT_FAILURE);
					}
					if(argp[2] != '\0') {
						strcpy(password, &argp[2]);
					} else {
						if(i == argc - 1) {
							error("no arg for -p");
							exit(EXIT_FAILURE);
						} else {
							strcpy(
								password,
								argv[++i]);
						}
					}
					break;

				case 'q':	/* Quiet mode */
					quiet = TRUE;
					break;

				case 's':	/* Specified server */
					if(servername[0] != '\0') {
						error(
							"server specified more"
							" than once");
						exit(EXIT_FAILURE);
					}
					if(argp[2] != '\0') {
						strcpy(servername, &argp[2]);
					} else {
						if(i == argc - 1) {
							error("no arg for -s");
							exit(EXIT_FAILURE);
						} else {
							strcpy(
								servername,
								argv[++i]);
						}
					}
					break;

				case 'u':	/* Specified username */
					if(username[0] != '\0') {
						error(
							"username specified"
							" more than once");
						exit(EXIT_FAILURE);
					}
					if(argp[2] != '\0') {
						strcpy(username, &argp[2]);
					} else {
						if(i == argc - 1) {
							error("no arg for -u");
							exit(EXIT_FAILURE);
						} else {
							strcpy(
								username,
								argv[++i]);
						}
					}
					break;

				case 'v':	/* Verbose - display progress */
					verbose = TRUE;
					break;

				case '\0':
					error("missing flag after '-'");
					exit(EXIT_FAILURE);

				default:
					error("invalid flag '%c'", argp[1]);
					exit(EXIT_FAILURE);
			}
		} else {
			add_file(argp);
		}
	}

	if(temp_port[0] != '\0') {	/* Port specified */
		i = strtol(temp_port, NULL, 0);
		if(i == 0) {
			error("invalid value for port number");
			exit(EXIT_FAILURE);
		}
		port = i;
	}

	if(servername[0] == '\0') {
		error("server must be specified using -s");
		exit(EXIT_FAILURE);
	}

	if(domain[0] != '\0') {		/* ETRN wanted */
		if(head != (PFL) NULL) {
			error("cannot send mail at same time as ETRN");
			exit(EXIT_FAILURE);
		}
	}

	if(((username[0] != '\0') && (password[0] == '\0')) ||
	   ((username[0] == '\0') && (password[0] != '\0'))) {
		error("neither or both of username and password must be"
		      " specified");
		exit(EXIT_FAILURE);
	}

	fix_domain(servername);

	if(domain[0] == 0) {		/* Not ETRN */
		if(head == (PFL) NULL) {
			PCHAR dir = getenv(SMTPDIR);
			PCHAR temp;

			if(dir == (PCHAR) NULL) {
				error(
					"no files specified,"
					" and environment variable "
					SMTPDIR" not set");
				exit(EXIT_FAILURE);
			}

			temp = xmalloc(strlen(dir)+1);
			strcpy(temp, dir);
			add_directory(temp);
		}

		/* Exit if nothing to do */

		if(something(head) == FALSE) {
			if(verbose == TRUE)
				fprintf(stdout, "No mail to send\n");
			exit(EXIT_SUCCESS);
		}
	}

	/* Get the host name of this client; if not possible, set it to the
	   dotted address. */

	rc = gethostname(clientname, sizeof(clientname));
	if(rc != 0) {
		INADDR myaddr;

		myaddr.s_addr = htonl(gethostid());
		sprintf(clientname, "[%s]", inet_ntoa(myaddr));
	} else {
		fix_domain(clientname);
	}

	sockno = socket(PF_INET, SOCK_STREAM, 0);
	if(sockno == -1) {
		error("cannot create socket");
		exit(EXIT_FAILURE);
	}

	smtphost = gethostbyname(servername);
	if(smtphost == (PHOST) NULL) {
		if(isdigit(servername[0])) {
			server_addr = inet_addr(servername);
		} else {
			error(
				"cannot get address for SMTP server '%s'",
				servername);
			exit(EXIT_FAILURE);
		}
	} else {
		server_addr = *((u_long *) smtphost->h_addr);
	}

	memset(&server, 0, sizeof(server));
	server.sin_family = AF_INET;
	server.sin_addr.s_addr = server_addr;
	server.sin_port = htons(port);
	rc = connect(sockno, (PSOCKG) &server, sizeof(SOCK));
	if(rc == -1) {
		error("cannot connect to SMTP server '%s'", servername);
		exit(EXIT_FAILURE);
	}

	/* Start logging */

	log_type = LOGGING_SYSLOG;
	if(logname[0] != '\0') log_type = LOGGING_FILE;
	rc = open_log(log_type, logname, clientname, progname);
	if(rc != LOGERR_OK) {
		error(
		"logging initialisation failed - %s",
		rc == LOGERR_OPENFAIL ? "file open failed" :
					"internal log type failure");
		exit(EXIT_FAILURE);
	}

	log_connection(servername, quiet);

	/* Do the work */

	rc = client(
			sockno, head, clientname, verbose,
			username, password, domain);

	(VOID) close(sockno);
	close_log();

	return(rc == TRUE ? EXIT_SUCCESS : EXIT_FAILURE);
}


/*
 * Add a filename to the file list.
 *
 */

static VOID add_file(PCHAR name)
{	PFL temp = (PFL) xmalloc(sizeof(FL));

	temp->next = (PFL) NULL;
	temp->name = name;
	temp->isdir = FALSE;
	strcpy(temp->name, name);
	if(head == (PFL) NULL) {
		head = temp;
		tail = temp;
	} else {
		tail->next = temp;
		tail = temp;
	}
}


/*
 * Add a directory to the file list.
 *
 */

static VOID add_directory(PCHAR name)
{	PFL temp = (PFL) xmalloc(sizeof(FL));

	temp->next = (PFL) NULL;
	temp->name = name;
	temp->isdir = TRUE;
	strcpy(temp->name, name);
	if(head == (PFL) NULL) {
		head = temp;
		tail = temp;
	} else {
		tail->next = temp;
		tail = temp;
	}
}


/*
 * Check for a full domain name; if not present, add default domain name.
 *
 */

static VOID fix_domain(PCHAR name)
{	if(strchr(name, '.') == (PCHAR) NULL && _res.defdname[0] != '\0') {
		strcat(name, ".");
		strcat(name, _res.defdname);
	}
}


/*
 * Print message on standard error in printf style,
 * accompanied by program name.
 *
 */

VOID error(PCHAR mes, ...)
{	va_list ap;

	fprintf(stderr, "%s: ", progname);

	va_start(ap, mes);
	vfprintf(stderr, mes, ap);
	va_end(ap);

	fputc('\n', stderr);
}


/*
 * Log details of the connection to standard output (if not quiet mode)
 * and to the logfile.
 *
 */

static VOID log_connection(PCHAR servername, BOOL quiet)
{	time_t tod;
	CHAR timeinfo[35];
	CHAR buf[100];

	if(quiet == FALSE) {
		(VOID) time(&tod);
		(VOID) strftime(timeinfo, sizeof(timeinfo),
			"on %a %d %b %Y at %X %Z", localtime(&tod));
		sprintf(buf, "%s: connection to %s, %s",
			progname, servername, timeinfo);
			fprintf(stdout, "%s\n", buf);
	}

	sprintf(buf, "connection to %s", servername);
	dolog(LOG_INFO, buf);
}


/*
 * Output program usage information.
 *
 */

static VOID putusage(VOID)
{	PCHAR *p = (PCHAR *) helpinfo;
	PCHAR q;

	for(;;) {
		q = *p++;
		if(*q == '\0') break;

		fprintf(stderr, q, progname);
		fputc('\n', stderr);
	}
	fprintf(stderr, "\nThis is version %d.%d\n", VERSION, EDIT);
}


/*
 * Allocate memory using 'malloc'; terminate with a message
 * if allocation failed.
 *
 */

PVOID xmalloc(size_t size)
{	PVOID res;

	res = malloc(size);

	if(res == (PVOID) NULL)
		error("cannot allocate memory");

	return(res);
}

/*
 * End of file: smtp.c
 *
 */

