/*
 * File beu.c
 *
 * Book entry utility
 *
 * Main program
 *
 * Bob Eager   May 2010
 *
 */

/*
 * History:
 *
 *	1.0	Initial version.
 *	1.1	No automatic file creation - force with -f.
 *		Default file extension now '.in-progress'.
 *
 */

#include "beu.h"

#define	DEFAULT_CONFIG	"/usr/local/etc/beu.conf"
#define	DEFAULT_EXTEN	"in-progress"

/* Forward references */

static	VOID		main_loop(FILE *, PCONFIG, PBOOK);
static	VOID		putusage(VOID);

/* Local storage */

static	CONFIG		config;
static	PCHAR		progname;

/* Help text */

static	const	PCHAR helpinfo[] = {
"%s: Book entry utility",
"Synopsis: %s [options] bookshelf",
" Options:",
"    -c file      specify alternate configuration file",
"    -f           force creation of output file",
"    -h           display this help",
"    -o file      specify output file",
" ",
"Default configuration file is "DEFAULT_CONFIG,
"Default file extension is \"."DEFAULT_EXTEN"\"",
"Bookshelf is of the form AA99",
""
};


/*
 * Parse arguments and handle options.
 *
 */

INT main(int argc, char *argv[])
{	INT i, rc;
	PCHAR ofile = (PCHAR) NULL;
	CHAR oftxt[SHELF_LENGTH+1+sizeof(DEFAULT_EXTEN)];
	PCHAR conf = (PCHAR) NULL;
	PCHAR argp, p;
	FILE *ofp;
	BOOK book;
	PCHAR bookshelf = (PCHAR) NULL;
	BOOL force = FALSE;

	progname = strrchr(argv[0], '/');
	if(progname != (PCHAR) NULL)
		progname++;
	else
		progname = argv[0];
	p = strchr(progname, '.');
	if(p != (PCHAR) NULL) *p = '\0';

	/* Process input options */

	for(i = 1; i < argc; i++) {
		argp = argv[i];
		if(argp[0] == '-') {		/* Option */
			switch(argp[1]) {
				case 'c':	/* Config file */
					if(conf != (PCHAR) NULL) {
						error(
							"configuration file"
							" specified "
							"more than once");
						exit(EXIT_FAILURE);
					}
					if(argp[2] != '\0') {
						conf = (PCHAR) strdup(&argp[2]);
					} else {
						if(i == argc - 1) {
							error(
								"no arg for -c");
							exit(EXIT_FAILURE);
						} else {
								conf = (PCHAR)
								strdup(argv[++i]);
						}
					}
					break;

				case 'f':	/* Force file creation */
					force = TRUE;
					break;

				case 'h':	/* Display help */
					putusage();
					exit(EXIT_SUCCESS);

				case 'o':	/* Output file */
					if(ofile != (PCHAR) NULL) {
						error(
							"output file"
							" specified "
							"more than once");
						exit(EXIT_FAILURE);
					}
					if(argp[2] != '\0') {
						ofile = (PCHAR) strdup(&argp[2]);
					} else {
						if(i == argc - 1) {
							error(
								"no arg for -o");
							exit(EXIT_FAILURE);
						} else {
								ofile = (PCHAR)
								strdup(argv[++i]);
						}
					}
					break;

				case '\0':
					error("missing flag after '-'");
					exit(EXIT_FAILURE);

				default:
					error("invalid flag '%c'", argp[1]);
					exit(EXIT_FAILURE);
			}
		} else {
			if(bookshelf != (PCHAR) NULL) {
				error("bookshelf can only be specified once");
				exit(EXIT_FAILURE);
			}
			bookshelf = argp;
		}
	}
	if(bookshelf == (PCHAR) NULL) {
		error("no bookshelf has been specified");
		exit(EXIT_FAILURE);
	}
	if((strlen(bookshelf) != SHELF_LENGTH) ||
	   !isalpha(bookshelf[0]) ||
	   !isalpha(bookshelf[1]) ||
	   !isdigit(bookshelf[2]) ||
	   !isdigit(bookshelf[3])) {
		error("bookshelf specification is incorrect"
			" - should be like AA99");
		exit(EXIT_FAILURE);
	}
	strcpy(book.shelf, bookshelf);
	book.shelf[0] = toupper(book.shelf[0]);
	book.shelf[1] = toupper(book.shelf[1]);
	
	if(ofile == (PCHAR) NULL) {	/* use default */
		ofile = oftxt;
		strcpy(ofile, bookshelf);
		strcat(ofile, "."DEFAULT_EXTEN);
	}
	if(conf == (PCHAR) NULL) conf = DEFAULT_CONFIG;

	/* Read configuration */

	rc = read_config(conf, &config);
	if(rc != 0) {
		error(
			"%d configuration error%s",
			rc, rc == 1 ? "" : "s");
		exit(EXIT_FAILURE);
	}

	/* Check that output file already exists, unless -f used */

	if(force == FALSE) {
		ofp = fopen(ofile, "r");
		if (ofp == (FILE *) NULL) {
			error("output file %s does not exist", ofile);
			exit(EXIT_FAILURE);
		}
		fclose(ofp);
	}

	/* Open output file for appending */

	ofp = fopen(ofile, "a+");
	if(ofp == (FILE *) NULL) {
		error("cannot open output file %s", ofile);
		exit(EXIT_FAILURE);
	}

	/* Initialise the pseudo-ISBN index */

	if(index_init(&config) == FALSE) {
		error("cannot intialise pseudo ISBN index");
		exit(EXIT_FAILURE);
	}

	/* Do the main work */

	main_loop(ofp, &config, &book);

	/* Tidy up and exit */

	index_final();
	fclose(ofp);

	return(EXIT_SUCCESS);
}


/*
 * Carry out the main processing.
 *
 *	Inputs:
 *		ofp	output file pointer
 *		config	pointer to configuration structure
 *		book	book structure
 *
 *	Outputs:
 *		none
 *
 */

static VOID main_loop(FILE *ofp, PCONFIG config, PBOOK book)
{	INT i;
	BOOL res;

	initscr();			/* clear and initialise screen */
	refresh();			/* initial refresh after above */
	cbreak();			/* set input mode */
	noecho();			/* turn off character echo */
	nonl();				/* turn off newline output */
	intrflush(stdscr, FALSE);	/* no special interrupt processing */

	draw_initial_screen(book);	/* draw framework */

	for(;;) {			/* Process a book */
		res = get_book(config, book);	/* Get the input */
		if(res == FALSE) break;	/* No more books */

		/* Correct for no-author case */

		if(book->nauthors == 0) {	/* No authors! */
			book->nauthors = 1;
			strcpy(book->authors[0], AUTHOR_UNKNOWN);
		}

		/* Write out the record */

		for(i = 0; i < book->nauthors; i++) {
			fprintf(ofp,
				"\"M\",\"%s\",\"%s\",\"%s\",\"%s\"\n",
				book->isbn, book->shelf, book->authors[i], book->title);
		}
		fflush(ofp);		/* commit to disk */

		message("Book recorded OK", book);
		sleep(ACCEPT_DELAY);
		clear_message(book);
	}

	erase(); refresh();		/* clear the screen */
	echo();				/* input echo back on */
	nl();				/* newline processing back on */
	nocbreak();			/* restore input mode */
	endwin();			/* restore screen */
}


/*
 * Print message on standard error in printf style,
 * accompanied by program name.
 *
 */

VOID error(PCHAR mes, ...)
{	va_list ap;

	fprintf(stderr, "%s: ", progname);

	va_start(ap, mes);
	vfprintf(stderr, mes, ap);
	va_end(ap);

	fputc('\n', stderr);
}


/*
 * Output program usage information.
 *
 */

static VOID putusage(VOID)
{	PCHAR *p = (PCHAR *) helpinfo;
	PCHAR q;

	for(;;) {
		q = *p++;
		if(*q == '\0') break;

		fprintf(stderr, q, progname);
		fputc('\n', stderr);
	}
	fprintf(stderr, "\nThis is version %d.%d\n", VERSION, EDIT);
}

/*
 * End of file: beu.c
 *
 */
