/*
 * File: index.c
 *
 * Book entry utility
 *
 * Index number generator - returns unique pseudo ISBNs
 *
 * Bob Eager   May 2010
 *
 */

#include "beu.h"
#include <errno.h>
#include <sys/stat.h>
#include <libgen.h>

/* Definitions */

#define	BITESIZE		(10)	/* Number of preallocated numbers */
#define	INDEXD_PERM		(S_IRWXU|S_IRWXG|S_IROTH|S_ISTXT)
#define	INDEX_PERM		(S_IRUSR|S_IWUSR|S_IRGRP|S_IWGRP|S_IROTH|S_IWOTH)

/* Local storage */

static	ULONG	cache_base;		/* base of values in cache */
static	INT	cache_remaining;	/* remaining numbers in cache */
static	PCHAR	index_file;		/* Name of index file */

/* Forward references */

static	VOID	refill_cache(VOID);


/*
 * Initialise the index generator, creating the index file if it does not
 * exist.
 *
 *	Inputs:
 *		config	pointer to configuration structure
 *
 *	Outputs:
 *		TRUE	initialised OK
 *		FALSE	initialisation failed
 *
 */

BOOL index_init(PCONFIG config)
{	INT rc;
	INT temp;
	PCHAR p;
	BOOL first = TRUE;
	FILE *index_fp;

	/* Try to open the count file; if this fails, make sure
	   the path exists, and try again. */

	for(;;) {
		index_fp = fopen(config->countfile, "r");
		if(index_fp != (FILE *) NULL) break;	/* opened OK */
		if(first == FALSE) return(FALSE);
		first = FALSE;
		rc = errno;
		if(rc == ENOENT) {	/* create path */
			p = dirname(config->countfile);
			(VOID) mkdir(p, INDEXD_PERM);
			chmod(p, INDEXD_PERM);
			index_fp = fopen(config->countfile, "w");
			if(index_fp == (FILE *) NULL) return(FALSE);
			fclose(index_fp);
			chmod(config->countfile, INDEX_PERM);
		}
	}

	/* Insert an initial value */

	rc = fscanf(index_fp, "%d", &temp);
	if(rc <= 0) {			/* empty file */
		fclose(index_fp);
		index_fp = fopen(config->countfile, "w");
		fputs("0\n", index_fp);
	}
	fclose(index_fp);
	index_file = config->countfile;	/* save for later */

	/* Initialise the number cache to 'empty' */

	cache_remaining = 0;

	return(TRUE);
}


/*
 * Finalise the index generator.
 * This writes back unused, cached values to the index file.
 *
 *	Inputs:
 *		none
 *
 *	Outputs:
 *		none
 *
 */

VOID index_final(VOID)
{	FILE *index_fp;

	if(cache_remaining == 0) return;	/* nothing to do */

	index_fp = fopen(index_file, "w");
	if(index_fp == (FILE *) NULL) return;
	fprintf(index_fp, "%ld\n", cache_base + BITESIZE - cache_remaining);
	fclose(index_fp);
}


/*
 * Return the next index value.
 *
 *	Inputs:
 *		none
 *	Outputs:
 *		index value
 *
 */

ULONG index_next(VOID)
{	ULONG res;

	if(cache_remaining <= 0) refill_cache();

	res = cache_base + BITESIZE - cache_remaining + 1;
	cache_remaining--;

	return(res);
}


/*
 * Refill the cache. This involves reading the current base value from
 * the index file, allocating BITESIZE values, and updating the index
 * file with the new base value. That value will be reduced on exit, to
 * reclaim unused values.
 *
 *	Inputs:
 *		none
 *	Outputs:
 *		none
 *
 */

static VOID refill_cache(VOID)
{	FILE *index_fp;

	index_fp = fopen(index_file, "r");
	if(index_fp == (FILE *) NULL) return;
	(void) fscanf(index_fp, "%ld", &cache_base);
	fclose(index_fp);

	cache_remaining = BITESIZE;

	index_fp = fopen(index_file, "w");
	if(index_fp == (FILE *) NULL) return;
	fprintf(index_fp, "%ld\n", cache_base + BITESIZE);
	fclose(index_fp);
}
 
/*
 * End of file: index.c
 *
 */
