/*
 * File: input.c
 *
 * Book entry utility
 *
 * Screen input handler
 *
 * Bob Eager   May 2010
 *
 */

#include "beu.h"
#include <errno.h>
#include <sys/stat.h>

/* Constants */

#define	HELP_LINE	" TAB=next field"\
			"    F5=clear field"\
			"    F6=clear all"\
			"    F8=accept"\
			"    F10=exit"

/* Definitions of relevant keys */

#define	DEL		0x7f
#define	TAB		'\t'
#define	CR		'\r'
#define	CLEAR		KEY_F(5)
#define	CLEAR_ALL	KEY_F(6)
#define	ACCEPT		KEY_F(8)
#define	EXIT		KEY_F(10)
#define	AUTO_ISBN	KEY_F(12)

/* Character checking tables */

static	CHAR	isbn_check[] =
	"0123456789Xx-";
static	CHAR	title_check[] =
	"ABCDEFGHIJKLMNOPQRSTUVWXYZ"
	"abcdefghijklmnopqrstuvwxyz"
	"0123456789"
	" .!&()-_+=[]{}:;'#?/";
static	CHAR	author_check[] =
	"ABCDEFGHIJKLMNOPQRSTUVWXYZ"
	"abcdefghijklmnopqrstuvwxyz"
	"0123456789"
	" -.&()'";

/* Forward references */

static	VOID		centre(INT, PCHAR);
static	BOOL		check_input(PBOOK);
static	VOID		clear_input_fields(PBOOK);
static	SHORT		get_book_inner(PCONFIG, PBOOK);
static	SHORT		get_field(PWINDOW, PCHAR, PCHAR, PBOOK);
static	BOOL		key_in(SHORT, PCHAR);

/*
 * Get details for one book.
 * On entry, the basic screen framework has been drawn.
 *
 *	Inputs:
 *		config	pointer to configuration structure
 *		book	pointer to book structure
 *
 *	Outputs:
 *		TRUE	valid book being returned in 'book'
 *		FALSE	exit requested
 *
 */

BOOL get_book(PCONFIG config, PBOOK book)
{	SHORT key;
	BOOL accept;

	clear_input_fields(book);	/* display empty input fields */
	for(;;) {			/* loop until valid data or exit */
		accept = FALSE;
		key = get_book_inner(config, book);
		switch(key) {
			case EXIT:
				return(FALSE);	/* exit requested */

			case CLEAR_ALL:		/* clear all fields */
				clear_input_fields(book);
				continue;	/* try input again */

			case ACCEPT:		/* accept input */
				accept = check_input(book);
				break;

			default:
				fatal("Unexpected key in get_book", book);
		}
		if(accept == TRUE) break;
	}
	return(TRUE);
}


/*
 * Get details for one book.
 * On entry, the basic screen framework has been drawn.
 * This is an inner function that does not do cleanup and validation.
 *
 *	Inputs:
 *		config	pointer to configuration structure
 *		book	pointer to book structure
 *
 *	Outputs:
 *		TRUE	valid book being returned in 'book'
 *		FALSE	exit requested
 *
 */

static SHORT get_book_inner(PCONFIG config, PBOOK book)
{	INT i;
	SHORT key;
	BOOL done;
	CHAR temp[MAXTITLE];

	temp[0] = '\0';
	book->title[0] = '\0';
	for(i = 0; i < NAUTHORS; i++) book->authors[i][0] = '\0';

	/* Get and check the ISBN */

	for(done = FALSE; done == FALSE;) {
		key = get_field(book->w_isbn, temp, isbn_check, book);
		switch(key) {
			case EXIT:	/* exit requested */
			case CLEAR_ALL:	/* clear all fields */
				return(key);

			case CLEAR:	/* clear this field */
				wclear(book->w_isbn);
				continue;

			case ACCEPT:	/* accept input and return */
				if(check_isbn(temp, book->isbn, book) == TRUE)
					return(key);
				message("ISBN is not valid", book);
				wclear(book->w_isbn);
				continue;

			case AUTO_ISBN:	/* fill in pseudo ISBN */
				next_pseudo_isbn(temp);
				strcpy(book->isbn, temp);
				wclear(book->w_isbn);
				mvwaddstr(book->w_isbn, 0, 0, temp);
				touchwin(stdscr);
				refresh();
				done = TRUE;
				break;

			case TAB:	/* next field */
				if(check_isbn(temp, book->isbn, book) == TRUE) {
					done = TRUE;
					break;
				}
				message("ISBN is not valid", book);
				wclear(book->w_isbn);
				continue;

			default:
				fatal("bad field terminator in get_book", book);
		}
	}

	/* Get the title */

continue_input:

	for(done = FALSE; done == FALSE;) {
		key = get_field(book->w_title, book->title, title_check, book);
		switch(key) {
			case EXIT:	/* exit requested */
			case CLEAR_ALL:	/* clear all fields */
				return(key);

			case CLEAR:	/* clear this field */
				wclear(book->w_title);
				continue;

			case ACCEPT:	/* accept input and return */
				return(key);

			case TAB:	/* next field */
				done = TRUE;
				break;

			default:
				fatal("bad field terminator in get_book", book);
		}
	}

	/* Get the authors */

	for(i = 0; i < NAUTHORS; i++) {
		for(done = FALSE; done == FALSE;) {
			key = get_field(book->w_authors[i],
				book->authors[i], author_check, book);
			switch(key) {
				case EXIT:	/* exit requested */
				case CLEAR_ALL:	/* clear all fields */
					return(key);

				case CLEAR:	/* clear this field */
					wclear(book->w_authors[i]);
					continue;

				case CR:
				case ACCEPT:	/* accept input and return */
					return(key);

				case TAB:	/* next field */
					done = TRUE;
					break;

				default:
					fatal("bad field terminator in get_book", book);
			}
		}
	}

	/* If we get here, it will be because TAB has been pressed in the
	   last author field. Go round again! */

	goto continue_input;
}


/*
 * Check that all input has been entered.
 *
 *	Inputs:
 *		book	pointer to book structure
 *
 *	Outputs:
 *		TRUE if all input present and correct
 *		FALSE otherwise
 *
 */

static BOOL check_input(PBOOK book)
{	INT i;
	INT first_empty, last_full;

	/* Check that the ISBN is non-null, although it should already
	   have been checked. */

	if(book->isbn[0] == '\0') {
		message("Missing ISBN", book);
		return(FALSE);
	}

	/* Check that the title is non-null. */

	if(book->title[0] == '\0') {
		message("Missing title", book);
		return(FALSE);
	}

	/* Check that at least one author field is non-null.
	   At the same time, compact the authors into a dense set,
	   and count them, recording their number. */

	for(;;) {
		first_empty = last_full = -1;
		book->nauthors = 0;
		for(i = 0; i < NAUTHORS; i++) {
			if(book->authors[i][0] != '\0') {
				last_full = i;
				book->nauthors++;
			} else {
				if(first_empty == -1)
					first_empty = i;
			}
		}
		if(first_empty == -1) break;	/* no gaps */
		if(last_full == -1) break;	/* no authors anyway */
		if(last_full == book->nauthors - 1) break;
		strcpy(book->authors[first_empty], book->authors[last_full]);
		book->authors[last_full][0] = '\0';
		/* now go round again, until compacted */
	}

	book->valid = TRUE;		/* Mark data valid for later */
	return(TRUE);
}


/*
 * Get input from a field.
 *
 *	Inputs:
 *		win	window for input
 *		buf	buffer for field contents
 *		check	character check table
 *		book	pointer to book structure
 *
 *	Outputs:
 *		key that terminated input
 *
 */

static SHORT get_field(PWINDOW win, PCHAR buf, PCHAR check, PBOOK book)
{	SHORT key;
	INT p = 0;
	INT q = 0;

	wbkgd(win, INPUT_ATTR);
	touchwin(stdscr);
	refresh();

	for(;;) {
		key = mvwgetch(win, 0, p);
		clear_message(book);
		switch(key) {
			case DEL:		/* Delete (backarrow) */
			case KEY_BACKSPACE:	/* ^H */
			case KEY_DC:		/* DEL */
				if(p != 0) {
					p--; q--;
					mvwaddch(win, 0, p, ' ');
					touchwin(stdscr);
					refresh();
				}
				continue;

			case AUTO_ISBN:
				if(win != book->w_isbn) break;
				buf[q] = '\0';
				wbkgd(win, FIELD_ATTR);
				touchwin(stdscr);
				refresh();
				return(key);

			case CR:
				key = TAB;
			case ACCEPT:
			case TAB:
			case CLEAR_ALL:
			case EXIT:
				buf[q] = '\0';
				wbkgd(win, FIELD_ATTR);
				touchwin(stdscr);
				refresh();
				return(key);

			case CLEAR:	/* clear this field */
				wclear(win);
				q = 0;
				buf[0] = '\0';
				p = 0;
				continue;
		}
		if(key_in(key, check) == FALSE) {
			beep();
			continue;
		}

		mvwaddch(win, 0, p++, key);
		buf[q++] = key;
		touchwin(stdscr);
		refresh();
	}
}


/*
 * Clear all of the input fields.
 *
 *	Inputs:
 *		book	pointer to book structure
 *
 *	Outputs:
 *		none
 *
 */

static VOID clear_input_fields(PBOOK book)
{	INT i;

	wbkgd(book->w_isbn, FIELD_ATTR);
	wclear(book->w_isbn);
	wbkgd(book->w_title, FIELD_ATTR);
	wclear(book->w_title);
	for(i = 0; i < NAUTHORS; i++) {
		wbkgd(book->w_authors[i], FIELD_ATTR);
		wclear(book->w_authors[i]);
	}
	touchwin(stdscr);
	refresh();
}


/*
 * Draw the initial screen. This also creates the subwindows for the various
 * fields.
 *
 *	Inputs:
 *		book	pointer to book structure
 *
 *	Outputs:
 *		none
 *
 */

VOID draw_initial_screen(PBOOK book)
{	CHAR temp[100];
	INT i, x;

	/* do the fixed areas */

	attron(BANNER_ATTR);
	centre(BANNER_Y, "BOOK DATA INPUT");
	attroff(BANNER_ATTR);
	sprintf(temp, "Shelf %s", book->shelf);
	attron(SHELF_ATTR);
	centre(SHELF_Y, temp);
	attroff(SHELF_ATTR);
	mvaddstr(ISBN_Y, ISBN_HELP_X, "<-- F12 to insert pseudo ISBN");
	mvaddstr(HELP_Y, 1, HELP_LINE);
	border(0, 0, 0, 0, 0, 0, 0, 0);
	curs_set(0);
	refresh();

	/* now the input and output fields, which are all subwindows */

	x = FIELD_X - strlen(ISBN_LABEL) - 2;	/* ISBN field */
	mvaddstr(ISBN_Y, x, ISBN_LABEL);
	book->w_isbn = subwin(stdscr, 1, ISBN_LENGTH, ISBN_Y, FIELD_X);
	keypad(book->w_isbn, TRUE);		/* enable keypad processing */

	x = FIELD_X - strlen(TITLE_LABEL) - 2;	/* Title field */
	mvaddstr(TITLE_Y, x, TITLE_LABEL);
	book->w_title = subwin(stdscr, 1, TITLE_LENGTH, TITLE_Y, FIELD_X);
	keypad(book->w_title, TRUE);		/* enable keypad processing */

	for(i = 0; i < NAUTHORS; i++) {		/* Author fields */
		sprintf(temp, AUTHOR_LABEL, i+1);
		x = FIELD_X - strlen(temp) - 2;
		mvaddstr(AUTHOR_Y + i*2, x, temp);
		book->w_authors[i] =
			subwin(stdscr, 1, AUTHOR_LENGTH, AUTHOR_Y + i*2,
				FIELD_X);
		keypad(book->w_authors[i], TRUE);/* enable keypad processing */
	}
	book->w_message = subwin(stdscr, 1, MESSAGE_LENGTH, MESSAGE_Y, MESSAGE_X);
	wattron(book->w_message, MESSAGE_ATTR);

	refresh();
}


/*
 * Display centred text on the screen.
 *
 *	Inputs:
 *		y	y position
 *		s	text
 *
 *	Outputs:
 *		none
 *
 */

static VOID centre(INT y, PCHAR s)
{	INT l = strlen(s);

	mvaddstr(y, (WIDTH - l)/2, s);
}


/*
 * Check whether a key appears in a check table.
 *
 *	Inputs:
 *		key	key to be checked
 *		check	pointer to check table
 *
 *	Outputs:
 *		TRUE	key is in table
 *		FALSE	key is not in table
 *
 */

static BOOL key_in(SHORT key, PCHAR check)
{	PCHAR p = memchr(check, key, strlen(check));

	if(p == (PCHAR) NULL) return(FALSE);

	return(TRUE);
}

/*
 * End of file: input.c
 *
 */
