/*
 * File: isbn.c
 *
 * Book entry utility
 *
 * ISBN handling
 *
 * Bob Eager   May 2010
 *
 */

#include "beu.h"

#define	BOOKLAND	"978"		/* Prefix for 13 digit ISBNs */
#define	PSEUDOBOOKLAND	"999"		/* Prefix for pseudo ISBNs */

/* Forward references */

static	INT		check10(PCHAR);
static	INT		check13(PCHAR);


/*
 * Check an ISBN. 10 digit ISBNs are validated, and converted to 13
 * digit ISBNs. 13 digit ISBNs are just validated.
 * Dashes are removed, if present.
 *
 *	Inputs:
 *		isbn	isbn for checking
 *		buf	buffer in which to return correct ISBN string
 *		book	pointer to book structure
 *
 *	Outputs:
 *		TRUE if ISBN is valid
 *		FALSE if ISBN is not valid
 *
 */

BOOL check_isbn(PCHAR isbn, PCHAR buf, PBOOK book)
{	INT c, l;
	PCHAR p, q;
	CHAR s[100];

	/* Strip out dashes */

	p = &isbn[0];
	q = &s[0];
	for(;;) {
		c = *p++;
		if(c == '\0') {
			*q = '\0';
			break;
		}
		if(c == '-') continue;
		*q++ = c;
	}
	l = strlen(s);			/* get length for checking */

	/* The putative ISBN is now in 's'. We know that by now
	   it contains only digits and possibly an X. */

	if(l == 10) {			/* 10 digit ISBN */
		s[9] = toupper(s[9]);	/* In case of X as check digit */
		if(s[9] == check10(s)) {

			/* Convert to 13 digit ISBN */

			s[9] = '\0';		/* Lose check digit */
			strcpy(buf, BOOKLAND);	/* Add leading Bookland code */
			strcat(buf, s);
			buf[12] = check13(buf);	/* Add new check digit */
			buf[13] = '\0';		/* Terminate */
			return(TRUE);
		} else return(FALSE);
	}

	if(l == 13) {			/* 13 digit ISBN */
		if(s[12] == check13(s)) {
			strcpy(buf, s);
			return(TRUE);
		} else return(FALSE);
	}
	return(FALSE);
}


/*
 * Generate a pseudo ISBN.
 * This is like a real 13 digit ISBN, but with a pseudo prefix. A valid
 * check digit is generated. The rest of the number is from a monotonically
 * increasing sequence; the next available value is kept in the sequence
 * counter ('index') file.
 *
 *	Inputs:
 *		s	buffer to receive ISBN
 *	Outputs:
 *		ISBN in s
 *
 */

VOID next_pseudo_isbn(PCHAR s)
{	ULONG index = index_next();	/* get raw number */
	INT check;

	sprintf(s, PSEUDOBOOKLAND"%09ld", index);
					/* generate ISBN without check digit */
	check = check13(s);
	sprintf(s, PSEUDOBOOKLAND"%09ld%c", index, check);
}


/*
 * Return the check digit for a 10 digit ISBN.
 *
 * Inputs:
 *	isbn	ISBN as a string (may or may not have a check digit).
 *		This ISBN is assumed to be otherwise valid.
 *
 * Outputs:
 *	single character for check digit
 *
 */

static INT check10(PCHAR isbn)
{	INT i, c;
	INT check = 0;

	for(i = 0; i < 9; i++) {
		c = isbn[i] - '0';
		check = check + c*(i+1);
	}
	c = check%11;
	c = (c == 10) ? 'X': c + '0';
	return(c);
}


/*
 * Return the check digit for a 13 digit ISBN.
 *
 * Inputs:
 *	isbn	ISBN as a string (may or may not have a check digit).
 *		This ISBN is assumed to be otherwise valid.
 *
 * Outputs:
 *	single character for check digit
 *
 */

static INT check13(PCHAR isbn)
{	INT i, c;
	INT check = 0;
	static const INT wt[12] = { 1,3,1,3,1,3,1,3,1,3,1,3 };

	for(i = 0; i < 12; i++) {
		c = isbn[i] - '0';
		check = check + (c * wt[i]);
	}
	c = check%10;		/* Remainder */
	if(c != 0) c = 10 - c;
	c = c + '0';		/* Convert to character */
	return(c);
}

/*
 * End of file: isbn.c
 *
 */
