/*
 * File: config.c
 *
 * Book entry utility
 *
 * Configuration file handler
 *
 * Bob Eager   May 2010
 *
 */

#include "beu.h"
#include "confcmds.h"

/* Forward references */

static	VOID	config_error(INT, PCHAR, ...);
static	INT	getcmd(PCHAR);


/*
 * Read and parse the configuration file specified by 'configfile'.
 *
 *	Inputs:
 *		configfile	pointer to name of config file
 *		config		pointer to config structure
 *
 *	Outputs:
 *		Number of errors encountered.
 *		Any error messages have already been issued.
 *
 * The configuration information is returned in the structure 'config'.
 *
 */

INT read_config(PCHAR configfile, PCONFIG config)
{	INT errors = 0;
	INT line = 0;
	PCHAR p, q, r, s, temp;
	FILE *fp;
	CHAR buf[MAXLINE];

	/* Set defaults */

	memset(config, 0, sizeof(CONFIG));
	config->countfile = (PCHAR) NULL;

	fp = fopen(configfile, "r");
	if(fp == (FILE *) NULL) {
		config_error(0, "cannot open configuration file %s", configfile);
		return(++errors);
	}

	for(;;) {
		p = fgets(buf, MAXLINE, fp);
		if(p == (PCHAR) NULL) break;
		temp = p + strlen(p) - 1;	/* Point to last character */
		if(*temp == '\n') *temp = '\0';	/* Remove any newline */
		line++;

		p = strchr(buf, '#');		/* Strip comments */
		if(p != (PCHAR) NULL) *p = '\0';

		p = strtok(buf, " \t");
		q = strtok(NULL, " \t");
		r = strtok(NULL, " \t");
		s = strtok(NULL, " \t");

		/* Skip non-information lines */

		if((p == (PCHAR) NULL) ||	/* No tokens */
		   (*p == '\n'))		/* Empty line */
			continue;

		switch(getcmd(p)) {
			case CMD_COUNTFILE:
				if(config->countfile != (PCHAR) NULL) {
					config_error(
						line,
						"COUNTFILE specified more"
						" than once");
					errors++;
					continue;
				}
				if(r != (PCHAR) NULL) {
					config_error(
						line,
						"syntax error (extra on end)");
					errors++;
					continue;
				}
				if(q == (PCHAR) NULL) {
					config_error(
						line,
						"no filename after "
						"COUNTFILE command");
					errors++;
					break;
				}
				p = malloc(strlen(q)+1);
				if(p == (PCHAR) NULL) {
					config_error(
						0,
						"cannot allocate memory");
					errors++;
					break;
				}
				strcpy(p, q);
				config->countfile = p;
				break;

			default:
				config_error(
					line,
					"unrecognised command '%s'", p);
				errors++;
				break;
		}
	}

	fclose (fp);

	if(config->countfile == (PCHAR) NULL) {
		config_error(0, "a COUNTFILE entry must be given");
		errors++;
	}

	return(errors);
}


/*
 * Check command for validity, and return command code; case is immaterial.
 *
 *	Inputs:
 *		s	pointer to command
 *
 *	Outputs
 *		CMD_BAD if command not recognised.
 *		command code if command recognised
 *
 */

static INT getcmd(PCHAR s)
{	INT i;

	for(i = 0; ; i++) {
		if(cmdtab[i].cmdcode == CMD_BAD) return(CMD_BAD);
		if(strcasecmp(s, cmdtab[i].cmdname) == 0) break;
	}

	return(cmdtab[i].cmdcode);
}


/*
 * Output configuration error message to standard error in printf style.
 *
 *	Inputs:
 *		line	line number of error
 *		mes	message for output
 *		...	optional message arguments
 *
 */

static VOID config_error(INT line, PCHAR mes, ...)
{	va_list ap;
	CHAR buf[MAXLOG];

	va_start(ap, mes);
	vsprintf(buf, mes, ap);
	va_end(ap);

	if(line == 0)
		error("config: %s", buf);
	else
		error("config: line %d: %s", line, buf);
}

/*
 * End of file: config.c
 *
 */
