/*
 * File: config.c
 *
 * Book lookup utility
 *
 * Configuration file handler
 *
 * Bob Eager   April 2014
 *
 */

#include "blu.h"
#include "confcmds.h"

/* Forward references */

static	VOID	config_error(INT, PCHAR, ...);
static	INT	getcmd(PCHAR);


/*
 * Read and parse the configuration file specified by 'configfile'.
 *
 * Returns:
 *	Number of errors encountered.
 *	Any error messages have already been issued.
 *
 * The configuration information is returned in the structure 'config'.
 *
 */

INT read_config(PCHAR configfile, PCONFIG config)
{	INT errors = 0;
	INT line = 0;
	PCHAR p, q, r, s, temp;
	FILE *fp;
	CHAR buf[MAXLINE];

	/* Set defaults */

	memset(config, 0, sizeof(CONFIG));
	config->db_url = (PCHAR) NULL;
	config->access_key = (PCHAR) NULL;
	config->cuecat = FALSE;

	fp = fopen(configfile, "r");
	if(fp == (FILE *) NULL) {
		config_error(0, "cannot open configuration file %s", configfile);
		return(++errors);
	}

	for(;;) {
		p = fgets(buf, MAXLINE, fp);
		if(p == (PCHAR) NULL) break;
		temp = p + strlen(p) - 1;	/* Point to last character */
		if(*temp == '\n') *temp = '\0';	/* Remove any newline */
		line++;

		p = strchr(buf, '#');		/* Strip comments */
		if(p != (PCHAR) NULL) *p = '\0';

		p = strtok(buf, " \t");
		q = strtok(NULL, " \t");
		r = strtok(NULL, " \t");
		s = strtok(NULL, " \t");

		/* Skip non-information lines */

		if((p == (PCHAR) NULL) ||	/* No tokens */
		   (*p == '\n'))		/* Empty line */
			continue;

		switch(getcmd(p)) {
			case CMD_DB_URL:
				if(config->db_url != (PCHAR) NULL) {
					config_error(
						line,
						"DB_URL specified more"
						" than once");
					errors++;
					continue;
				}
				if(r != (PCHAR) NULL) {
					config_error(
						line,
						"syntax error (extra on end)");
					errors++;
					continue;
				}
				if(q == (PCHAR) NULL) {
					config_error(
						line,
						"no URL after "
						"DB_URL command");
					errors++;
					break;
				}
				strlwr(q);
				if(strncmp(
						q,
						HTTP_SCHEME,
						strlen(HTTP_SCHEME)) != 0) {
					config_error(
						line,
						"bad scheme in DB_URL"
						" value");
					errors++;
					break;
				}
				p = malloc(strlen(q)+1);
				if(p == (PCHAR) NULL) {
					config_error(
						0,
						"cannot allocate memory");
					errors++;
					break;
				}
				strcpy(p, q);
				config->db_url = p;
				break;

			case CMD_ACCESS_KEY:
				if(config->access_key != (PCHAR) NULL) {
					config_error(
						line,
						"ACCESS_KEY specified more"
						" than once");
					errors++;
					continue;
				}
				if(r != (PCHAR) NULL) {
					config_error(
						line,
						"syntax error (extra on end)");
					errors++;
					continue;
				}
				if(q == (PCHAR) NULL) {
					config_error(
						line,
						"no key value after "
						"ACCESS_KEY command");
					errors++;
					break;
				}
				p = malloc(strlen(q)+1);
				if(p == (PCHAR) NULL) {
					config_error(
						0,
						"cannot allocate memory");
					errors++;
					break;
				}
				strcpy(p, q);
				config->access_key = p;
				break;

			case CMD_CUECAT:	
				if(r != (PCHAR) NULL) {
					config_error(
						line,
						"syntax error (extra on end)");
					errors++;
					continue;
				}
				if(q == (PCHAR) NULL) {
					config_error(
						line,
						"no value after "
						"CUECAT command");
					errors++;
					break;
				}
				strlwr(q);
				if(strcmp(q, "no") == 0) {
					config->cuecat = FALSE;
					break;
				}
				if(strcmp(q, "yes") == 0) {
					config->cuecat = TRUE;
					break;
				}
				config_error(
					line,
					"bad value '%s' after"
					" CUECAT command", q);
				errors++;
				break;

			default:
				config_error(
					line,
					"unrecognised command '%s'", p);
				errors++;
				break;
		}
	}

	fclose (fp);

	if(config->db_url == (PCHAR) NULL) {
		config_error(0, "a DB_URL entry must be given");
		errors++;
	}

	if(config->access_key == (PCHAR) NULL) {
		config_error(0, "an ACCESS_KEY entry must be given");
		errors++;
	}

	return(errors);
}


/*
 * Check command in 's' for validity, and return command code.
 * Case is immaterial.
 *
 * Returns CMD_BAD if command not recognised.
 *
 */

static INT getcmd(PCHAR s)
{	INT i;

	for(i = 0; ; i++) {
		if(cmdtab[i].cmdcode == CMD_BAD) return(CMD_BAD);
		if(strcasecmp(s, cmdtab[i].cmdname) == 0) break;
	}

	return(cmdtab[i].cmdcode);
}


/*
 * Output configuration error message to standard error in printf style.
 *
 */

static VOID config_error(INT line, PCHAR mes, ...)
{	va_list ap;
	CHAR buf[MAXLOG];

	va_start(ap, mes);
	vsprintf(buf, mes, ap);
	va_end(ap);

	if(line == 0)
		error("config: %s", buf);
	else
		error("config: line %d: %s", line, buf);
}

/*
 * End of file: config.c
 *
 */
