/*
 * File blu.c
 *
 * Book lookup utility
 *
 * Main program
 *
 * Bob Eager   April 2014
 *
 */

/*
 * History:
 *
 *	1.0	Initial version.
 *	1.1	Ported to FreeBSD.
 *	1.2	No automatic file creation - force with -f.
 *		Default file extension now '.in-progress'.
 *		Added -q option.
 *	1.3	Fixed select problem (wrong nfds) for FreeBSD 8.3 and later.
 *		Fixed truncated reason text in HTTP response.
 *
 */

#include "blu.h"

#define	DEFAULT_CONFIG	"/usr/local/etc/blu.conf"
#define	DEFAULT_EXTEN	"in-progress"
#define	TOTAL_RES	"<BookList total_results=\""

/* Forward references */

static	VOID		main_loop(FILE *, PCONFIG, PBOOK);
static	VOID		putusage(VOID);

/* Local storage */

static	CONFIG		config;
static	PCHAR		progname;

/* Help text */

static	const	PCHAR helpinfo[] = {
"%s: Book lookup utility",
"Synopsis: %s [options] bookshelf",
" Options:",
"    -c file      specify alternate configuration file",
"    -d           enable debug output",
"    -f           force creation of output file",
"    -h           display this help",
"    -o file      specify output file",
"    -q           quiet (no prompting)",
"    -v           verbose; display progress",
" ",
"Default configuration file is "DEFAULT_CONFIG,
"Default file extension is \"."DEFAULT_EXTEN"\"",
"Bookshelf is of the form AA99",
"Terminate input with a line consisting of a single dot.",
""
};


/*
 * Parse arguments and handle options.
 *
 */

INT main(int argc, char *argv[])
{	INT i, rc;
	PCHAR ofile = (PCHAR) NULL;
	CHAR oftxt[BOOKSHELF_LENGTH+1+sizeof(DEFAULT_EXTEN)];
	PCHAR conf = (PCHAR) NULL;
	PCHAR argp, p;
	FILE *ofp;
	BOOK book;
	PCHAR bookshelf = (PCHAR) NULL;
	BOOL force = FALSE;

	progname = strrchr(argv[0], '/');
	if(progname != (PCHAR) NULL)
		progname++;
	else
		progname = argv[0];
	p = strchr(progname, '.');
	if(p != (PCHAR) NULL) *p = '\0';
	strlwr(progname);

	/* Process input options */

	book.debug = FALSE;
	book.verbose = FALSE;
	book.quiet = FALSE;

	for(i = 1; i < argc; i++) {
		argp = argv[i];
		if(argp[0] == '-') {		/* Option */
			switch(argp[1]) {
				case 'c':	/* Config file */
					if(conf != (PCHAR) NULL) {
						error(
							"configuration file"
							" specified "
							"more than once");
						exit(EXIT_FAILURE);
					}
					if(argp[2] != '\0') {
						conf = (PCHAR) strdup(&argp[2]);
					} else {
						if(i == argc - 1) {
							error(
								"no arg for -c");
							exit(EXIT_FAILURE);
						} else {
								conf = (PCHAR)
								strdup(argv[++i]);
						}
					}
					break;

				case 'd':	/* Debug - display trace */
					book.debug = TRUE;
					break;

				case 'f':	/* Force file creation */
					force = TRUE;
					break;

				case 'h':	/* Display help */
					putusage();
					exit(EXIT_SUCCESS);

				case 'o':	/* Output file */
					if(ofile != (PCHAR) NULL) {
						error(
							"output file"
							" specified "
							"more than once");
						exit(EXIT_FAILURE);
					}
					if(argp[2] != '\0') {
						ofile = (PCHAR) strdup(&argp[2]);
					} else {
						if(i == argc - 1) {
							error(
								"no arg for -o");
							exit(EXIT_FAILURE);
						} else {
								ofile = (PCHAR)
								strdup(argv[++i]);
						}
					}
					break;

				case 'q':	/* Quiet */
					book.quiet = TRUE;
					break;

				case 'v':	/* Verbose - display progress */
					book.verbose = TRUE;
					break;

				case '\0':
					error("missing flag after '-'");
					exit(EXIT_FAILURE);

				default:
					error("invalid flag '%c'", argp[1]);
					exit(EXIT_FAILURE);
			}
		} else {
			if(bookshelf != (PCHAR) NULL) {
				error("bookshelf can only be specified once");
				exit(EXIT_FAILURE);
			}
			bookshelf = argp;
		}
	}

	/* Process and validate bookshelf specification */

	if(bookshelf == (PCHAR) NULL) {
		error("no bookshelf has been specified");
		exit(EXIT_FAILURE);
	}
	if((strlen(bookshelf) != BOOKSHELF_LENGTH) ||
	   !isalpha(bookshelf[0]) ||
	   !isalpha(bookshelf[1]) ||
	   !isdigit(bookshelf[2]) ||
	   !isdigit(bookshelf[3])) {
		error("bookshelf specification is incorrect");
		exit(EXIT_FAILURE);
	}
	strcpy(book.shelf, bookshelf);
	book.shelf[0] = toupper(book.shelf[0]);
	book.shelf[1] = toupper(book.shelf[1]);
	
	if(ofile == (PCHAR) NULL) {	/* use default */
		ofile = oftxt;
		strcpy(ofile, bookshelf);
		strcat(ofile, "."DEFAULT_EXTEN);
	}
	if(conf == (PCHAR) NULL) conf = DEFAULT_CONFIG;

	/* Read configuration */

	rc = read_config(conf, &config);
	if(rc != 0) {
		error(
			"%d configuration error%s",
			rc, rc == 1 ? "" : "s");
		exit(EXIT_FAILURE);
	}

	/* Check that output file already exists, unless -f used */

	if(force == FALSE) {
		ofp = fopen(ofile, "r");
		if (ofp == (FILE *) NULL) {
			error("output file %s does not exist", ofile);
			exit(EXIT_FAILURE);
		}
		fclose(ofp);
	}

	/* Open output file for appending */

	ofp = fopen(ofile, "a+");
	if(ofp == (FILE *) NULL) {
		error("cannot open output file %s", ofile);
		exit(EXIT_FAILURE);
	}

	/* Do the main work */

	main_loop(ofp, &config, &book);

	/* Tidy up and exit */

	fclose(ofp);

	return(EXIT_SUCCESS);
}


/*
 * Carry out the main processing.
 *
 *	Inputs:
 *		ofp	output file pointer
 *		config	pointer to configuration structure
 *		book	book structure
 *
 *	Outputs:
 *		none
 *
 */

#define	YBUFSIZE	80

static VOID main_loop(FILE *ofp, PCONFIG config, PBOOK book)
{	INT i, len;
	BOOL res;
	PCHAR p, q;
	CHAR buf[ISBNSIZE+1];
	CHAR ybuf[YBUFSIZE+1];

	for(;;) {			/* Process an ISBN */
		if(book->quiet == FALSE)
			fprintf(stdout, "Enter next ISBN, dot to finish...\n");
		res = get_isbn(buf, config, book);
		if(res == FALSE) break;	/* No more ISBNs */
		if(buf[0] == '\0') {
			error("this is not a valid ISBN");
			continue;
		}
		strcpy(book->isbn, buf);
		if(book->quiet == FALSE)
			fprintf(stdout, "Looking it up....\n");
		p = lookup_isbn(buf, config, &len);
		if(p == (PCHAR) NULL) {
			fprintf(stderr, "Lookup failed!\n");
			continue;
		}			/* Failed, but try again */

		/* Check number of matches */

		q = strstr(p, TOTAL_RES);
		if(q == (PCHAR) NULL) {
			error("\aNo results for %s\n", buf);
			lookup_free(p);
			continue;
		}
		q += strlen(TOTAL_RES);
		i = atoi(q);		/* Get number of results */
		if(i == 0) {
			error("\aNo results for %s\n", buf);
			lookup_free(p);
			continue;
		}
		if(i != 1) {
			error("\aMultiple results for %s\n", buf);
			lookup_free(p);
			continue;
		}
		if(book->verbose == TRUE)
			fprintf(stderr, "Looked up %s OK\n", buf);

		extract_book(p, len, book);	/* Extract data */
		lookup_free(p);			/* Free the data buffer */
		if(book->valid == FALSE) continue;

		/* Correct for no-author case */

		if(book->nauthors == 0) {	/* No authors! */
			book->nauthors = 1;
			strcpy(book->authors[0], AUTHOR_UNKNOWN);
		}

		/* Write out the record */

		/* Tell the user what we have */

		fprintf(stdout, "ISBN:\t\t%s\n", book->isbn);
		fprintf(stdout, "Title:\t\t%s\n", book->title);
		for(i = 0; i < book->nauthors; i++) {
			fprintf(stdout, "Author %d:\t%s\n",
					i+1, book->authors[i]);
		}

		/* Check it is OK */

		fprintf(stdout, "Is this OK [Y]? ");
		fflush(stdout);
		q = fgets(ybuf, YBUFSIZE, stdin);
		if(q == (PCHAR) NULL) break;	/* end of file */
		fpurge(stdin);
		if(ybuf[0] == '\n') strcpy(ybuf, "Y\n");
		if(toupper(ybuf[0]) != 'Y') continue;

		/* OK, dump the details to file */

		for(i = 0; i < book->nauthors; i++) {
			fprintf(ofp,
				"\"S\",\"%s\",\"%s\",\"%s\",\"%s\"\n",
				buf, book->shelf, book->authors[i], book->title);
		}
		fflush(ofp);		/* commit to disk */
		fprintf(stdout, "Recorded\n");
	}
}


/*
 * Print message on standard error in printf style,
 * accompanied by program name.
 *
 */

VOID error(PCHAR mes, ...)
{	va_list ap;

	fprintf(stderr, "%s: ", progname);

	va_start(ap, mes);
	vfprintf(stderr, mes, ap);
	va_end(ap);

	fputc('\n', stderr);
}


/*
 * Output program usage information.
 *
 */

static VOID putusage(VOID)
{	PCHAR *p = (PCHAR *) helpinfo;
	PCHAR q;

	for(;;) {
		q = *p++;
		if(*q == '\0') break;

		fprintf(stderr, q, progname);
		fputc('\n', stderr);
	}
	fprintf(stderr, "\nThis is version %d.%d\n", VERSION, EDIT);
}


/*
 * Log a message.
 *
 */

VOID dolog(PCHAR mes, ...)
{	va_list ap;

	fprintf(stderr, "%s: ", progname);

	va_start(ap, mes);
	vfprintf(stderr, mes, ap);
	va_end(ap);

	fputc('\n', stderr);
}


/*
 * Convert a string to lower case, in situ.
 *
 */

VOID strlwr(PCHAR s)
{	PCHAR p = s;
	INT c;

	while(*p != '\0') {
		c = tolower(*p);
		*p++ = c;
	}
}

/*
 * End of file: blu.c
 *
 */
