/*
 * File: cuecat.c
 *
 * Book lookup utility
 *
 * CueCat handling and decoding
 *
 * Bob Eager   April 2014
 *
 */

#include "blu.h"

#define	CCHDR		"\x38\x44\xc4\xb8"	/* Header string */
#define	CCXOR		0x43			/* XOR decrypt value */

/* Forward references */

static	PCHAR		debase64(PCHAR, PCHAR);

/*
 * Get input from the CueCat. This is connected via the keyboard.
 *
 *	Inputs:
 *		buf	input buffer
 *		max	length of input buffer
 *		debug	debug output flag
 *
 *	Outputs:
 *		pointer to buf, or NULL if error
 *
 */

PCHAR cuecat_gets(PCHAR buf, INT max, BOOL debug)
{	INT i, l, xtra;
	static CHAR s[MAXLINE+1];
	PCHAR p;

	p = fgets(s, max, stdin);
	if(p == (PCHAR) NULL) return(p);

	/* Handle the override case. The first character is '!', and
	   the rest of the input is plain text. */

	if((p[0] == '!') || (p[0] == '=')) {
		strcpy(buf, p+1);	/* Lose the '!' or '=' */
		return(buf);
	}

	/* The input buffer is terminated by a newline, and has
	   four fields separated by periods. These are:
		1)	CueCat header - always: 0x38, 0x44, 0xc4, 0xb8
			(is this still here?)
		2)	CueCat serial number - not needed
		3)	Barcode type - not needed
		4)	Barcode value
	   All values are encoded and encrypted. First decode them
	   (this is a modified base64 encoding), then decrypt (this is
	   a simple XOR).
	*/

	l = strlen(s);
	if(s[l-1] == '\n') {		/* Strip the newline */
		s[l-1] = '\0';
	}

	p = strtok(s, ".");		/* Lose the header */
	p = strtok(NULL, ".");		/* Lose the serial number */
	if(p == (PCHAR) NULL) return(p);
#if 0
	p = strtok(NULL, ".");		/* Lose the barcode type */
	if(p == (PCHAR) NULL) return(p);
#endif
	p = strtok(NULL, ".");		/* Point 'p' to the barcode value */
	if(p == (PCHAR) NULL) return(p);

	xtra = 4 - strlen(p)%4;
	if(xtra == 4) xtra = 0;
	switch(l) {
		case 3:	strcat(p, "=");
		case 2:	strcat(p, "=");
		case 1:	strcat(p, "=");
	}
	p = debase64(p, buf);		/* Decode base 64 encoding */
	l = strlen(buf);
	for(i = 0; i < l; i++) buf[i] ^= CCXOR;
					/* Decrypt */
	if(debug == TRUE)
		fprintf(stdout, "Scanned: %s\n", buf);

	return(p);
}


/*
 * Notes on modified base 64 encoding
 * ----------------------------------
 *
 * Strings are encoded as follows.
 * The input string is split into triples, each 24 bits in size (3 octets);
 * The earliest octet is stored as the most significant one.
 * To encode the triple, the 24 bit value is taken as four 6 bit values,
 * starting again at the most significant end. Each 6 bit value is encoded
 * by using an array of encoding characters, and simply indexing that.
 * Any terminating, incomplete triple is encoded with enough characters
 * to describe all of the actual characters in the partial triple.
 * The string is then padded with '=' to make its length a multiple of four.
 *
 */

static	const	CHAR dctab[] = {
/*   0 */ 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
/*   8 */ 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
/*  16 */ 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
/*  24 */ 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
/*  32 */ 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
/*  40 */ 0x00, 0x00, 0x00, 0x3e, 0x00, 0x3f, 0x00, 0x00,
/*  48 */ 0x34, 0x35, 0x36, 0x37, 0x38, 0x39, 0x3a, 0x3b,
/*  56 */ 0x3c, 0x3d, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
/*  64 */ 0x00, 0x1a, 0x1b, 0x1c, 0x1d, 0x1e, 0x1f, 0x20,
/*  72 */ 0x21, 0x22, 0x23, 0x24, 0x25, 0x26, 0x27, 0x28,
/*  80 */ 0x29, 0x2a, 0x2b, 0x2c, 0x2d, 0x2e, 0x2f, 0x30,
/*  88 */ 0x31, 0x32, 0x33, 0x00, 0x00, 0x00, 0x00, 0x00,
/*  96 */ 0x00, 0x00, 0x01, 0x02, 0x03, 0x04, 0x05, 0x06,
/* 104 */ 0x07, 0x08, 0x09, 0x0a, 0x0b, 0x0c, 0x0d, 0x0e,
/* 112 */ 0x0f, 0x10, 0x11, 0x12, 0x13, 0x14, 0x15, 0x16,
/* 120 */ 0x17, 0x18, 0x19, 0x00, 0x00, 0x00, 0x00, 0x00
};

/*
 * Decode a string from modified base 64.
 *
 * For convenience, returns a pointer to the decoded string.
 *
 */

static PCHAR debase64(PCHAR in, PCHAR out)
{	INT i, j;
	INT size = strlen(in);
	INT triple = 0;
	INT n = 0;
	INT noctets = 0;

	for(i = 0; i < size; i+=4) {
		noctets = 0;
		j = in[i];
		if(j == '=') break;
		noctets = 1;
		triple = dctab[j] << 18;
		j = in[i+1];
		if(j == '=') break;
		noctets = 2;
		triple = triple | (dctab[j] << 12);
		j = in[i+2];
		if(j == '=') break;
		noctets = 3;
		triple = triple | (dctab[j] << 6);
		j = in[i+3];
		if(j == '=') break;
		triple = triple | dctab[j];
		/* Have 3 octets in 'triple' here */
		out[n++] = triple >> 16;
		out[n++] = (triple >> 8) & 0xff;
		out[n++] = triple & 0xff;
		noctets = 0;
	}

	switch(noctets) {
		case 3:
			out[n++] = triple >> 16;
			out[n++] = (triple >> 8) & 0xff;
			out[n++] = triple & 0xff;
			break;

		case 2:
			out[n++] = triple >> 16;
			out[n++] = (triple >> 8) & 0xff;
			break;

		case 1:
			out[n++] = triple >> 16;
			break;

		case 0:
			break;
	}
	out[n] = '\0';			/* Add terminator */

	return(out);
}

/*
 * End of file: cuecat.c
 *
 */
