/*
 * File: isbn.c
 *
 * Book lookup utility
 *
 * ISBN handling
 *
 * Bob Eager   April 2014
 *
 */

#include "blu.h"

#define	BOOKLAND	"978"

/* Forward references */

static	INT		check10(PCHAR);
static	INT		check13(PCHAR);


/*
 * Read an ISBN. 10 digit ISBNs are validated, and converted to 13
 * digit ISBNs. 13 digit ISBNs are just validated.
 * Dashes are removed, if present.
 *
 *	Inputs:
 *		buf	buffer in which to return ISBN string.
 *		config	pointer to configuration structure
 *		book	pointer to book structure
 *
 *	Outputs:
 *		TRUE if read an ISBN
 *		FALSE if end of file, or line containing a single dot was
 *		read.
 *
 */

BOOL get_isbn(PCHAR buf, PCONFIG config, PBOOK book)
{	INT i, l, c;
	CHAR temp[MAXLINE], s[MAXLINE];
	PCHAR p, q;

	/* Read a line and strip off the terminating newline.
	   A line consisting of a single dot indicates no more data.
	*/

	if(config->cuecat == TRUE ) {	/* Use the CueCat */
		p = cuecat_gets(temp, MAXLINE, book->debug);
		if(p == (PCHAR) NULL) {	/* Error */
			if(book->quiet == FALSE)
				fprintf(stdout, "\a\a\a");
			fpurge(stdin);
			fflush(stdout);
			return(FALSE);
		} else {			/* Make 'OK' beep */
			if(strlen(p) > 13) {	/* Real CueCat input */
				if(book->quiet == FALSE)
					fprintf(stdout, "\a");
				fflush(stdout);
			}
		}
	} else {
		p = fgets(temp, MAXLINE, stdin);
		if(p == (PCHAR) NULL) return(FALSE);
	}
	l = strlen(temp);
	if(temp[l-1] == '\n') temp[--l] = '\0';	/* Strip newline */
	if((l == 1) && (temp[0] == '.'))
		return(FALSE);		/* Finished */

	/* Strip out white space and dashes */

	q = &s[0];
	for(;;) {
		c = *p++;
		if(c == '\0') {
			*q = '\0';
			break;
		}
		if((c == ' ') || (c == '\t') || (c == '-')) continue;
		*q++ = c;
	}
	l = strlen(s);			/* restore length */
	if(l > 13) l = 13;		/* handle overrun */
	s[l] = '\0';

	/* The putative ISBN is now in 's'. It may contain bad characters,
	   but the validation will in any case fix that. */

	buf[0] = '\0';			/* Error if unchanged */
	for(i = 0; i < l - 1; i++) {	/* Check all digits */
		if(!isdigit(s[i]))
			return(TRUE);
	}
	if(l == 10) {			/* 10 digit ISBN */
		s[9] = toupper(s[9]);
		if(s[9] == check10(s)) {

			/* Convert to 13 digit ISBN */

			s[9] = '\0';		/* Lose check digit */
			strcpy(buf, BOOKLAND);	/* Add leading Bookland code */
			strcat(buf, s);
			buf[12] = check13(buf);	/* Add new check digit */
			buf[13] = '\0';		/* Terminate */
		}
	}

	if(l == 13) {			/* 13 digit ISBN */
		if(s[12] == check13(s))
			strcpy(buf, s);
	}
	return(TRUE);
}


/*
 * Return the check digit for a 10 digit ISBN.
 *
 * Inputs:
 *	isbn	ISBN as a string (may or may not have a check digit).
 *		This ISBN is assumed to be otherwise valid.
 *
 * Outputs:
 *	single character for check digit
 *
 */

static INT check10(PCHAR isbn)
{	INT i, c;
	INT check = 0;

	for(i = 0; i < 9; i++) {
		c = isbn[i] - '0';
		check = check + c*(i+1);
	}
	c = check%11;
	c = (c == 10) ? 'X': c + '0';
	return(c);
}


/*
 * Return the check digit for a 13 digit ISBN.
 *
 * Inputs:
 *	isbn	ISBN as a string (may or may not have a check digit).
 *		This ISBN is assumed to be otherwise valid.
 *
 * Outputs:
 *	single character for check digit
 *
 */

static INT check13(PCHAR isbn)
{	INT i, c;
	INT check = 0;
	static const INT wt[12] = { 1,3,1,3,1,3,1,3,1,3,1,3 };

	for(i = 0; i < 12; i++) {
		c = isbn[i] - '0';
		check = check + (c * wt[i]);
	}
	c = check%10;		/* Remainder */
	if(c != 0) c = 10 - c;
	c = c + '0';		/* Convert to character */
	return(c);
}

/*
 * End of file: isbn.c
 *
 */
