/*
 * File: bwmon.c
 *
 * Bandwidth monitor
 *
 * Bob Eager   May 2016
 *
 */

#include "bwmon.h"
#ifdef HAVE_LIBPQ
#include <libpq-fe.h>
#endif

/* Forward references */

static	VOID		alloc_error(VOID);
static	VOID		commit_data(time_t);
static	VOID		credit(PSTATISTICS, const struct ip *);
static	PIPDATA		find_ip(UINT);
static	INT		fork2(VOID);
static	VOID		init_webpage(PCHAR, CHAR);
static	VOID		makepidfile(pid_t);
static	VOID		packetcallback(u_char *, const struct pcap_pkthdr *,
			const u_char *);
static	VOID		rcdf_load(FILE *);
static	VOID		rcdf_position_stream(FILE *);
static	BOOL		rcdf_test(PCHAR);
static	VOID		recover_data_from_cdf(VOID);
static	VOID		set_child_config(INT);
static	VOID		signal_handler(INT);
static	VOID		store_ipdata_in_cdf(IPDATA[]);
static 	INT		write_out_webpages(time_t);
static	VOID		usage(VOID);

/* Global storage */

CONFIG			config;
PIPDATASTORE		ipdatastore = (PIPDATASTORE) NULL;
INT			ftp_ports = 0;
INT			ftp_port[50];
INT			http_ports = 0;
INT			http_port[50];
INT			p2p_ports = 0;
INT			p2p_port[50];
UINT			subnet_count = 0;
SUBNETDATA		subnet_table[MAX_SUBNETS];

/* Local storage */

static	PCHAR		config_file = DEFAULT_CONFIG_FILE;
static	UINT		graphintervalcount;
static	time_t		interval_start;
static	INT		ip_offset;
static	UINT		ipcount = 0;
static	PIPDATA		iptable;
static	BOOL		parent = TRUE;
static	pcap_t		*pd;
static	INT		rotate_logs = 0;
static	pid_t		workerchildpids[NR_WORKER_CHILDS];

/* Tables */

static	const DOUBLE	range[] = {
RANGE1, RANGE2, RANGE3, RANGE4
};

static	const ULONGLONG interval[] = {
INTERVAL1, INTERVAL2, INTERVAL3, INTERVAL4
};

static	const INT default_ftp_port[] = {
20,21
};

static	const INT default_http_port[] = {
80, 443
};

static	const INT default_p2p_port[] = {
1044, 1045,		/* Direct File Express */
1214,			/* Grokster, Kazaa, Morpheus */
4661, 4662, 4665,	/* eDonkey 2000 */
5190,			/* Song Spy */
5500, 5501, 5502, 5503,	/* Hotline Connect */
6346, 6347,		/* Gnutella Engine */
6666, 6667,		/* Yoink */
7788,			/* BuddyShare */
8888,			/* AudioGnome, OpenNap, Swaptor */
8889,			/* AudioGnome, OpenNap */
28864, 28865		/* hotComm */
};

/* External references */

extern	INT		bmconf_parse(VOID);
extern	FILE		*bmconf_in;

/* Help text */

static	const PCHAR helpinfo[] = {
	"%s: Network traffic bandwidth monitor",
	"Synopsis: %s [options] [file...]",
	" Options:",
	"       -c file   specifiy alternate config file",
	"       -D        do not become a daemon",
	"       -h        this message",
	"       -l        list available devices",
	""
};

/* Startup code */

INT main(INT argc, PCHAR *argv)
{	INT i;
	BOOL daemon = TRUE;
	BOOL listdevs = FALSE;
	pcap_if_t *devices;
	struct bpf_program fcode;
	PUCHAR pcap_userdata = (PUCHAR) NULL;
	CHAR errbuf[PCAP_ERRBUF_SIZE];

	/* Set up configuration defaults */

	config.dev = (PCHAR) NULL;
	config.filter = "ip";
	config.max_ips = DEFAULT_MAXIPS;
	config.skip_intervals = DEFAULT_INTERVALS;
	config.graph_cutoff = DEFAULT_CUTOFF;
	config.top = DEFAULT_TOP;
	config.promisc = TRUE;
	config.graph = TRUE;
	config.output_cdf = FALSE;
	config.recover_cdf = FALSE;
	config.meta_refresh = DEFAULT_REFRESH;
	config.output_database = FALSE;
	config.db_connect_string = (PCHAR) NULL;
	config.sensor_id = "unset";

	/* Start syslogging */

	openlog(PROGNAME, LOG_CONS, LOG_DAEMON);

	/* Handle any arguments */

	for(i = 1; i < argc; i++) {
		if(argv[i][0] == '-') {
			switch(argv[i][1]) {
				case 'c':
					if(i+1 == argc) {
						usage();
						exit(EXIT_FAILURE);
					}
					config_file = argv[++i];
					break;

				case 'D':
					daemon = FALSE;
					break;

				case 'l':
					listdevs = TRUE;
			 		break;

				case 'h':
					usage();
					exit(EXIT_SUCCESS);

				default:
					usage();
					exit(EXIT_FAILURE);
			}
		}
	}

	/* Read the configuration file */

	bmconf_in = fopen(config_file, "r");
	if(bmconf_in == (FILE *) NULL) {
		syslog(LOG_ERR, "cannot open %s", config_file);
		fprintf(stderr, PROGNAME": cannot open %s\n", config_file);
		exit(EXIT_FAILURE);
	}
	bmconf_parse();
	fclose(bmconf_in);

	/* Set up IP array; allow for the pseudo-IP used for totals */

	config.max_ips++;
	iptable = (PIPDATA) calloc(config.max_ips, sizeof(IPDATA));
	if(iptable == (PIPDATA) NULL) {
		syslog(LOG_ERR,
			PROGNAME": cannot allocate memory for IP table");
		fprintf(stderr,
			PROGNAME": cannot allocate memory for IP table\n");
		exit(EXIT_FAILURE);
	}

	if(ftp_ports == 0) {
		ftp_ports = sizeof(default_ftp_port)/sizeof(INT);
		for(i = 0; i < ftp_ports; i++)
			ftp_port[i] = default_ftp_port[i];
	}
	if(http_ports == 0) {
		http_ports = sizeof(default_http_port)/sizeof(INT);
		for(i = 0; i < http_ports; i++)
			http_port[i] = default_http_port[i];
	}
	if(p2p_ports == 0) {
		p2p_ports = sizeof(default_p2p_port)/sizeof(INT);
		for(i = 0; i < p2p_ports; i++)
			p2p_port[i] = default_p2p_port[i];
	}

	/* Enumerate network devices and set default if required. */

	pcap_findalldevs(&devices, errbuf);
	if(config.dev == (PCHAR) NULL && devices->name != (PCHAR) NULL)
		config.dev = strdup(devices->name);

	/* Handle -l flag and exit */

	if(listdevs == TRUE) {
		while(devices != NULL) {
			printf("Description: %s\n"
				"Name: \"%s\"\n"
				"\n",
				devices->description, devices->name);
			devices = devices->next;
		}
		exit(EXIT_SUCCESS);
	}

	/* At least one subnet must be specified. */

	if(subnet_count == 0) {
		syslog(LOG_ERR, "no subnets to monitor");
		fprintf(stderr, PROGNAME": no subnets to monitor\n");
		exit(EXIT_FAILURE);
	}

	/* Handle default value for htdocs path, and normalise it */

	if(config.htdocs == (PCHAR) NULL)
		config.htdocs = DEFAULT_HTDOCS;
	if(config.htdocs[strlen(config.htdocs)-1] != '/') {
		PCHAR s = (PCHAR) malloc(strlen(config.htdocs)+2);

		if(s == (PCHAR) NULL)
			alloc_error();
		strcpy(s, config.htdocs);
		strcat(s, "/");
		config.htdocs = s;
	}

	/* Handle default value for logs path, and normalise it */

	if(config.logs == (PCHAR) NULL)
		config.logs = DEFAULT_LOGS;

	if(config.logs[strlen(config.logs)-1] != '/') {
		PCHAR s = (PCHAR) malloc(strlen(config.logs)+2);

		if(s == (PCHAR) NULL)
			alloc_error();
		strcpy(s, config.logs);
		strcat(s, "/");
		config.logs = s;
	}

	/* Handle default value for PID file */

	if(config.pidfile == (PCHAR) NULL)
		config.pidfile = DEFAULT_PIDFILE;

	/* Set up graph placeholders */

	if(config.graph == TRUE) {
		init_webpage(INDEX_HTML,  '1');
		init_webpage(INDEX2_HTML, '2');
		init_webpage(INDEX3_HTML, '3');
		init_webpage(INDEX4_HTML, '4');
	}

	/* Detach from console if daemon */

	if(daemon == TRUE && fork2())
		exit(EXIT_SUCCESS);	/* parent exits here */

	/* Now we can store our PID for later signals */

	makepidfile(getpid());

	/* Initialise first (day graphing) process config */

	set_child_config(0);

	/* If necessary, fork processes for week, month and year graphing
	   and/or local data collection. */

	if(config.graph == TRUE || config.output_cdf == TRUE) {
		for(i = 0; i < NR_WORKER_CHILDS; i++) {
			workerchildpids[i] = fork();

			/* Initialise children and let them start doing work,
			 * while parent continues to fork children.
			 */

			if(workerchildpids[i] == 0) {	/* child */
				parent = FALSE;
				set_child_config(i+1);
				break;
			}

			if(workerchildpids[i] == -1) {	/* fork failed */
				syslog(
					LOG_ERR,
					"Failed to fork graphing child (%d)",
					i);
					continue;
			}
		}

		if(config.recover_cdf == TRUE)
			recover_data_from_cdf();
	}

	/* Get baseline time */

	interval_start = time(NULL);

	/* Open network device for monitoring */

	syslog(LOG_INFO, "(%c): opening %s", config.tag, config.dev);
	errbuf[0] = '\0';
	pd = pcap_open_live(
		config.dev,		/* device name */
		100,			/* max number of bytes to grab */
		config.promisc,		/* promiscuous mode */
		1000,			/* read timeout (ms) */
		errbuf);		/* warning/error information */
	if(pd == (pcap_t *) NULL) {	/* Failed to open device */
		syslog(LOG_ERR, "(%c) %s", config.tag, errbuf);
		exit(EXIT_FAILURE);
	}
	if(errbuf[0] != '\0') 		/* warning text present */
		syslog(LOG_WARNING, "(%c): %s", config.tag, errbuf);

	/* Compile the filter string and install it */

	{	CHAR buf[20];

		sprintf(buf, "Error (%c):", config.tag);

		if(pcap_compile(pd, &fcode, config.filter, 1, 0) < 0) {
			pcap_perror(pd, buf);
			fprintf(
				stderr,
				"(%c): malformed libpcap filter string in "
				"%s\n", config.tag, config_file);
			syslog(
				LOG_ERR,
				"(%c): malformed libpcap filter string in "
				"%s", config.tag, config_file);
			exit(EXIT_FAILURE);
		}
		if(pcap_setfilter(pd, &fcode) < 0)
        		pcap_perror(pd, buf);
	}

        /* Report the datalink type for completeness */

	switch(pcap_datalink(pd)) {
		default:
			syslog(
				LOG_INFO,
				"(%c): unknown datalink type,"
				" defaulting to Ethernet", config.tag);
			/* drop through */

		case DLT_EN10MB:
			syslog(
				LOG_INFO,
				"(%c): packet encoding: Ethernet", config.tag);
			ip_offset = sizeof(struct ether_header);
			break;
	}

	/* If we are now a daemon, lose the standard files as we cannot
	   use them. */

	if(daemon == TRUE) {
		fclose(stdin);
		fclose(stdout);
		fclose(stderr);
	}

	/* Install the signal handler; HUP for logfile rotatiom,
	   and TERM for cleanup. */

	signal(SIGHUP, signal_handler);
	signal(SIGTERM, signal_handler);

	/* If there is data in the datastore, draw some initial graphs */

	if(ipdatastore != (PIPDATASTORE) NULL) {
		syslog(LOG_INFO, "(%c): drawing initial graphs", config.tag);
		write_out_webpages(interval_start + config.interval);
	}

	/* Start main monitoring loop */

	if(pcap_loop(
		pd,			/* capture descriptor */
		-1,			/* loop forever */
		packetcallback,		/* callback function */
		pcap_userdata		/* unused */
		) < 0) {		/* failed - exit */
        	syslog(LOG_ERR,
			"(%c): pcap_loop: %s", config.tag, pcap_geterr(pd));
      		exit(EXIT_FAILURE);
        }
	pcap_close(pd);

	if(parent == TRUE)
		unlink(config.pidfile);

	exit(EXIT_SUCCESS);
}


/*
 * Packet capture callback function.
 *
 */

static VOID packetcallback(u_char *user, const struct pcap_pkthdr *h,
	const u_char *p)
{	UINT i;
	UINT caplen = h->caplen;
	UINT srcip, dstip;
	PIPDATA ptr_ipdata;
	const struct ip *ip;

	/* The packet's timestamp effectively gives us the current time,
	   which can be used to see if the current interval has completed.
	   Use this to write out this interval's data, and possibly kick
	   off a set of graphs. */

	if(h->ts.tv_sec > interval_start + config.interval) {
		graphintervalcount++;
		commit_data(interval_start + config.interval);
		ipcount = 0;			/* reset IP table */
		interval_start = h->ts.tv_sec;	/* start of next interval */
	}

	/* Only measuring IP size, so remove Ethernet header */

	caplen -= ip_offset;
	p += ip_offset;			/* point past the datalink header */
	ip = (const struct ip *) p;	/* point 'ip' at the IP header */
	if(ip->ip_v != IPVERSION) return;/* not an IP packet - ignore */

	/* Harvest the source and destination IP addresses */

	srcip = ntohl(*(PUINT) (&ip->ip_src));
	dstip = ntohl(*(PUINT) (&ip->ip_dst));

	/* Credit the packet to all appropriate monitored subnets.
	   Packets from a monitored subnet to a monitored subnet
	   will be credited to both IPs */

	for(i = 0; i < subnet_count; i++) {	/* scan possible subnets */
		if(subnet_table[i].ip == (srcip & subnet_table[i].mask)) {
			ptr_ipdata = find_ip(0);	/* totals */
			if(ptr_ipdata != (PIPDATA) NULL) /* may have no room */
				credit(&(ptr_ipdata->send), ip);

			ptr_ipdata = find_ip(srcip);	/* this IP */
			if(ptr_ipdata != (PIPDATA) NULL)
				credit(&(ptr_ipdata->send), ip);
		}

		if(subnet_table[i].ip == (dstip & subnet_table[i].mask)) {
			ptr_ipdata = find_ip(0);	/* totals */
			if(ptr_ipdata != (PIPDATA) NULL)
				credit(&(ptr_ipdata->receive), ip);

			ptr_ipdata = find_ip(dstip);	/* this IP */
			if(ptr_ipdata != (PIPDATA) NULL)
				credit(&(ptr_ipdata->receive), ip);
		}
	}
}


/*
 * Create a PID file in the usual place so that we can be easily signalled.
 *
 */

static VOID makepidfile(pid_t pid)
{	FILE *pidfile;

	pidfile = fopen(config.pidfile, "w");
	if(pidfile != (FILE *) NULL) {
		if(fprintf(pidfile, "%d\n", pid) == 0) {
			syslog(LOG_ERR,
				"failed to write '%d' to %s",
				pid, config.pidfile);
			fclose(pidfile);
			unlink(config.pidfile);
		} else {
			fclose(pidfile);
		}
	} else {
		syslog(LOG_ERR, "Could not open %s for write", config.pidfile);
	}
}


/*
 * Credit transferred bytes to an IP address.
 * 'stats' is a pointer to the particular counter to be updated, and 'ip'
 * points to the packet header.
 *
 */

static VOID credit(PSTATISTICS stats, const struct ip *ip)
{	INT i;
	ULONG size;
	USHORT sport, dport;
	const struct tcphdr *tcp;

	size = ntohs(ip->ip_len);	/* packet length */

	stats->total += size;		/* always increment total */
	switch(ip->ip_p) {		/* record protocol stats */
		case IPPROTO_TCP:
			tcp = (struct tcphdr *) (ip + 1);
					/* point beyond main IP header */
			tcp = (struct tcphdr *) (((char *)tcp) +
						 ((ip->ip_hl-5)*4));
					/* compensate for any IP options */
			stats->tcp += size;

			sport = ntohs(tcp->th_sport);
			dport = ntohs(tcp->th_dport);

			for(i = 0; i < http_ports; i++) {
				if(sport == http_port[i] ||
				   dport == http_port[i]) {
					stats->http += size;
					return;
				}
			}

			for(i = 0; i < ftp_ports; i++) {
				if(sport == ftp_port[i] ||
				   dport == ftp_port[i]) {
					stats->ftp += size;
					return;
				}
			}

			for(i = 0; i < p2p_ports; i++) {
				if(sport == p2p_port[i] ||
				   dport == p2p_port[i]) {
					stats->p2p += size;
					return;
				}
			}
			break;

		case IPPROTO_UDP:
			stats->udp += size;
			break;

		case IPPROTO_ICMP:
			stats->icmp += size;
			break;
	}
}


/*
 * Go through the RAM datastore and dump old data.
 *
 */

static VOID drop_old_data(LONG timestamp)
{	PIPDATASTORE datastore;
	PIPDATASTORE predatastore;
	PDATASTOREBLOCK deletedblock;

	predatastore = (PIPDATASTORE) NULL;
	datastore = ipdatastore;

	/* Scan the list */
	while(datastore != (PIPDATASTORE) NULL)	{
		/* If the first block is out of date, purge it;
		   if it is the only block, purge the node */
		while(datastore->firstblock->latesttimestamp <
			timestamp - config.range) {
			if((!datastore->firstblock->next) && predatastore) {
				/* There is no valid block of data for this IP,
				   so unlink the whole IP */
				predatastore->next = datastore->next;
						/* Unlink the node */
				free(datastore->firstblock->data);
				free(datastore->firstblock);
				free(datastore);
				datastore = predatastore->next;
				if(datastore == (PIPDATASTORE) NULL) return;
			} else
			if(!datastore->firstblock->next) {
				/* There is no valid block of data for this IP,
				   and we are the first IP, so do nothing */
				break;
			} else { /* Just unlink this block */
				deletedblock = datastore->firstblock;
				datastore->firstblock =	/* Unlink the block */
					datastore->firstblock->next;
				free(deletedblock->data);
				free(deletedblock);
			}
		} /* while timestamp */
		predatastore = datastore;
		datastore = datastore->next;
	} /* while over list */
}


/*
 * Put data into a PostGresSQL database
 *
 */

static VOID store_ipdata_in_postgresql(IPDATA incdata[])
{
#ifdef	HAVE_LIBPQ
	PIPDATA ipdata;
	UINT i;
	PSTATISTICS stats;
	PGresult *res;
	static PGconn *conn = (PGconn *) NULL;
	static CHAR sensor_id[50];
	const char *paramvalues[10];
	PCHAR sql1;
	PCHAR sql2;
	CHAR values[10][50];

	if(config.output_database != DB_PGSQL) return;

#if 0
	for(i = 0; i < 10; i++)
		paramvalues[i] = values[i];
#endif

	/* Initialise database if necessary */

	if(conn == (PGconn *) NULL) {	/* Connect to the database */
	    	conn = PQconnectdb(config.db_connect_string);
		if(PQstatus(conn) != CONNECTION_OK) {
			syslog(LOG_ERR,
				"Connection to database '%s' failed: %s",
				config.db_connect_string, PQerrorMessage(conn));
			PQfinish(conn);
			conn = (PGconn *) NULL;
			return;
		}

		strncpy(values[0], config.sensor_id, 50);
		res = PQexecParams(
			conn,
			"select sensor_id from sensors where sensor_name = $1;",
			1,	 /* one param */
			NULL,    /* let the backend deduce param type */
			paramvalues,
			NULL,    /* don't need param lengths since text */
			NULL,	 /* default to all text params */
			0);      /* ask for binary results */

		if(PQresultStatus(res) != PGRES_TUPLES_OK) {
			syslog(
				LOG_ERR,
				"PostgreSQL SELECT failed: %s",
				PQerrorMessage(conn));
			PQclear(res);
			PQfinish(conn);
			conn = (PGconn *) NULL;
			return;
		}

		if(PQntuples(res)) {
			strncpy(sensor_id, PQgetvalue(res, 0, 0), 50);
			PQclear(res);
		} else {	/* Insert new sensor id */
			PQclear(res);
			res = PQexecParams(
				conn,
				"insert into sensors"
				" (sensor_name, last_connection)"
				" VALUES ($1, now());",
				1,	 /* one param */
				NULL,	 /* let the backend deduce param type */
				paramvalues,
				NULL,	 /* don't need param lengths since text */
				NULL,	 /* default to all text params */
				0);	 /* ask for binary results */

			if(PQresultStatus(res) != PGRES_COMMAND_OK) {
				syslog(LOG_ERR, "PostgreSQL INSERT failed: %s",
					PQerrorMessage(conn));
				PQclear(res);
				PQfinish(conn);
				conn = (PGconn *) NULL;
				return;
			}
			PQclear(res);
			res = PQexecParams(
				conn,
				"select sensor_id from sensors"
				" where sensor_name = $1;",
				1,	 /* one param */
				NULL,    /* let the backend deduce param type */
				paramvalues,
				NULL,    /* don't need param lengths since text */
				NULL,	 /* default to all text params */
				0);      /* ask for binary results */

			if(PQresultStatus(res) != PGRES_TUPLES_OK) {
				syslog(LOG_ERR, "PostgreSQL SELECT failed: %s",
					PQerrorMessage(conn));
				PQclear(res);
				PQfinish(conn);
				conn = (PGconn *)NULL;
				return;
			}
			strncpy(sensor_id, PQgetvalue(res, 0, 0), 50);
			PQclear(res);
		}
	}

	/* Begin transaction */

	/* Perform inserts */

	res = PQexecParams(
		conn,
		"BEGIN;",
		0,	 /* zero param */
		NULL,    /* let the backend deduce param type */
		NULL,
		NULL,	 /* don't need param lengths since text */
		NULL,	 /* default to all text params */
		0);	 /* ask for binary results */

	if(PQresultStatus(res) != PGRES_COMMAND_OK) {
		syslog(LOG_ERR, "PostgreSQL BEGIN failed: %s",
			PQerrorMessage(conn));
		PQclear(res);
		PQfinish(conn);
		conn = (PGconn *) NULL;
		return;
	}
	PQclear(res);

	strncpy(values[0], sensor_id, 50);

	res = PQexecParams(
		conn,
		"update sensors set last_connection = now()"
		" where sensor_id = $1;",
		1,	 /* one param */
		NULL,    /* let the backend deduce param type */
		paramvalues,
		NULL,	 /* don't need param lengths since text */
		NULL,	 /* default to all text params */
		0);	 /* ask for binary results */

	if(PQresultStatus(res) != PGRES_COMMAND_OK) {
		syslog(LOG_ERR, "PostgreSQL UPDATE failed: %s",
			PQerrorMessage(conn));
		PQclear(res);
		PQfinish(conn);
		conn = (PGconn *) NULL;
		return;
	}
	PQclear(res);

	values[0][49] = '\0';
	snprintf(values[1], 50, "%llu", config.interval);
	for(i = 0; i < ipcount; i++) {
		ipdata = &incdata[i];
		if(ipdata->ip == 0) {
			/* This optimization allows us to quickly draw
			   totals graphs for a sensor */
			sql1 = "INSERT INTO bd_tx_total_log"
				" (sensor_id, sample_duration, ip, total,"
				" icmp, udp, tcp, ftp, http, p2p)"
				" VALUES($1, $2, $3, $4, $5, $6, $7, $8, $9,"
				" $10);";
			sql2 = "INSERT INTO bd_rx_total_log"
				" (sensor_id, sample_duration, ip, total,"
				" icmp, udp, tcp, ftp, http, p2p)"
				" VALUES($1, $2, $3, $4, $5, $6, $7, $8, $9,"
				" $10);";
		} else {
			sql1 = "INSERT INTO bd_tx_log"
				" (sensor_id, sample_duration, ip, total,"
				" icmp, udp, tcp, ftp, http, p2p)"
				" VALUES($1, $2, $3, $4, $5, $6, $7, $8, $9,"
				" $10);";
			sql2 = "INSERT INTO bd_rx_log"
				" (sensor_id, sample_duration, ip, total,"
				" icmp, udp, tcp, ftp, http, p2p)"
				" VALUES($1, $2, $3, $4, $5, $6, $7, $8, $9,"
				" $10);";
		}

		hostip_to_charip(ipdata->ip, values[2]);

		stats = &(ipdata->send);
		if(stats->total > 512) { /* Don't log empty sets */
			/* Log data in kilobytes */
			snprintf(values[3], 50, "%llu", (ULONGLONG)((((DOUBLE)stats->total)/1024.0) + 0.5));
			snprintf(values[4], 50, "%llu", (ULONGLONG)((((DOUBLE)stats->icmp)/1024.0) + 0.5));
			snprintf(values[5], 50, "%llu", (ULONGLONG)((((DOUBLE)stats->udp)/1024.0) + 0.5));
			snprintf(values[6], 50, "%llu", (ULONGLONG)((((DOUBLE)stats->tcp)/1024.0) + 0.5));
			snprintf(values[7], 50, "%llu", (ULONGLONG)((((DOUBLE)stats->ftp)/1024.0) + 0.5));
			snprintf(values[8], 50, "%llu", (ULONGLONG)((((DOUBLE)stats->http)/1024.0) + 0.5));
			snprintf(values[9], 50, "%llu", (ULONGLONG)((((DOUBLE)stats->p2p)/1024.0) + 0.5));

			res = PQexecParams(
				conn,
				sql1,
				10,	/* nine params */
				NULL,   /* let the backend deduce param type */
				paramvalues,
				NULL,    /* don't need param lengths since text */
				NULL,	 /* default to all text params */
				1);      /* ask for binary results */

			if(PQresultStatus(res) != PGRES_COMMAND_OK) {
				syslog(LOG_ERR,
					"PostgreSQL INSERT failed: %s",
					PQerrorMessage(conn));
				PQclear(res);
				PQfinish(conn);
				conn = (PGconn *) NULL;
				return;
			}
			PQclear(res);
		}
		stats = &(ipdata->receive);
		if(stats->total > 512) { /* Don't log empty sets */
			snprintf(values[3], 50, "%llu", (ULONGLONG)((((DOUBLE)stats->total)/1024.0) + 0.5));
			snprintf(values[4], 50, "%llu", (ULONGLONG)((((DOUBLE)stats->icmp)/1024.0) + 0.5));
			snprintf(values[5], 50, "%llu", (ULONGLONG)((((DOUBLE)stats->udp)/1024.0) + 0.5));
			snprintf(values[6], 50, "%llu", (ULONGLONG)((((DOUBLE)stats->tcp)/1024.0) + 0.5));
			snprintf(values[7], 50, "%llu", (ULONGLONG)((((DOUBLE)stats->ftp)/1024.0) + 0.5));
			snprintf(values[8], 50, "%llu", (ULONGLONG)((((DOUBLE)stats->http)/1024.0) + 0.5));
			snprintf(values[9], 50, "%llu", (ULONGLONG)((((DOUBLE)stats->p2p)/1024.0) + 0.5));

			res = PQexecParams(
				conn, sql2,
				10,       /* seven params */
				NULL,    /* let the backend deduce param type */
				paramvalues,
				NULL,    /* don't need param lengths since text */
				NULL,    /* default to all text params */
				1);      /* ask for binary results */

	 		if(PQresultStatus(res) != PGRES_COMMAND_OK) {
				syslog(LOG_ERR,
					"PostgreSQL INSERT failed: %s",
					PQerrorMessage(conn));
				PQclear(res);
				PQfinish(conn);
				conn = (PGconn *) NULL;
				return;
			}
			PQclear(res);
		}
	}

	/* Commit transaction */

	res = PQexecParams(
			conn,
			"COMMIT;",
			0,	/* zero params */
			NULL,	/* let the backend deduce param type */
			NULL,
			NULL,	/* don't need param lengths since text */
			NULL,	/* default to all text params */
			0);	/* ask for binary results */

	if(PQresultStatus(res) != PGRES_COMMAND_OK) {
		syslog(LOG_ERR, "PostgreSQL COMMIT failed: %s",
			PQerrorMessage(conn));
		PQclear(res);
		PQfinish(conn);
		conn = (PGconn *) NULL;
		return;
	}
	PQclear(res);
#else
	syslog(LOG_ERR,
		"PostgreSQL logging selected but PostgreSQL support"
		" is not compiled into binary.");
#endif
}


/*
 * Put data into a MySQL database
 *
 */

static VOID store_ipdata_in_mysql(IPDATA incdata[])
{
	syslog(LOG_ERR,
		"MySQL logging selected but MySQL support"
		" is not compiled into binary.");
	return;
}


/*
 * Store IP data in external database.
 *
 */

static VOID store_ipdata_in_database(IPDATA incdata[])
{	switch(config.output_database) {
		case DB_PGSQL:
			store_ipdata_in_postgresql(incdata);
			break;

		case DB_MYSQL:
			store_ipdata_in_mysql(incdata);
			break;

		default:
			;	/* ignore */
	}
}


/*
 * Store data into a CDF file.
 *
 */

static VOID store_ipdata_in_cdf(IPDATA incdata[])
{	PIPDATA ipdata;
	UINT i;
	ULONG l;
	FILE *cdf;
	PSTATISTICS stats;
	CHAR ipbuffer[50];
	CHAR temp[MAXPATH];
	CHAR logfile[MAXPATH];		/* log.X.0.cdf, X=tag */

	sprintf(temp, CDF_FILE, config.tag, 0);	/* build log.X.0.cdf */
	strcpy(logfile, config.logs);
	strcat(logfile, temp);		/* build full path */
   	cdf = fopen(logfile, "a");

	for(i = 0; i < ipcount; i++) {
		ipdata = &incdata[i];
		hostip_to_charip(ipdata->ip, ipbuffer);
		l = (ULONG) ipdata->timestamp;
		fprintf(cdf, "%s,%lu,", ipbuffer, l);
		stats = &(ipdata->send);
		fprintf(cdf, "%llu,%llu,%llu,%llu,%llu,%llu,%llu,",
			stats->total, stats->icmp, stats->udp,
			stats->tcp, stats->ftp, stats->http, stats->p2p);
		stats = &(ipdata->receive);
		fprintf(cdf, "%llu,%llu,%llu,%llu,%llu,%llu,%llu\n",
			stats->total, stats->icmp, stats->udp,
			stats->tcp, stats->ftp, stats->http, stats->p2p);
	}
	fclose(cdf);
}


/*
 * Output error message about memory allocation failure, and exit.
 *
 */

static VOID alloc_error(VOID)
{	syslog(LOG_ERR,	"(%c): could not allocate memory - exiting",
			config.tag);
	exit(EXIT_FAILURE);
}


/*
 * Dump the data into a formatted memory area, since we need to re-use
 * the IP table right now.
 *
 */

static VOID _store_ipdata_in_ram(PIPDATA ipdata)
{	PIPDATASTORE datastore;
	PDATASTOREBLOCK datastore_block;

	if(ipdatastore == (PIPDATASTORE) NULL) { /* need to create first entry */
		/* allocate datastore for this IP */
		ipdatastore = (PIPDATASTORE) malloc(sizeof(IPDATASTORE));
		if(ipdatastore == (PIPDATASTORE) NULL)
			alloc_error();
		ipdatastore->ip = ipdata->ip;
		ipdatastore->next = (PIPDATASTORE) NULL; /* end of chain */

		/* allocate its first block of storage */

		ipdatastore->firstblock =
			(PDATASTOREBLOCK) malloc(sizeof(DATASTOREBLOCK));
		if(ipdatastore->firstblock == (PDATASTOREBLOCK) NULL)
			alloc_error();
		ipdatastore->firstblock->latesttimestamp = 0;
		ipdatastore->firstblock->numentries = 0;
		ipdatastore->firstblock->data =
			(PIPDATA) calloc(IPDATAALLOCCHUNKS, sizeof(IPDATA));
		if(ipdatastore->firstblock->data == (PIPDATA) NULL)
			alloc_error();
		ipdatastore->firstblock->next = (PDATASTOREBLOCK) NULL;
	}

	datastore = ipdatastore;	/* Point to global start of list */

	while(datastore != (PIPDATASTORE) NULL) {
		if(datastore->ip == ipdata->ip) { /* we have the right store */
			datastore_block = datastore->firstblock;

			while(datastore_block != (PDATASTOREBLOCK) NULL) {
				if(datastore_block->numentries <
					IPDATAALLOCCHUNKS) { /* have a free spot */
					memcpy(&datastore_block->data[datastore_block->numentries],
						ipdata, sizeof(IPDATA));
					datastore_block->numentries++;
					datastore_block->latesttimestamp = ipdata->timestamp;
					return;
				} else {
					if(datastore_block->next == (PDATASTOREBLOCK) NULL) {
						/* there isn't another block,
						   add one */
						datastore_block->next = (PDATASTOREBLOCK) malloc(sizeof(DATASTOREBLOCK));
						if(datastore_block->next == (PDATASTOREBLOCK) NULL)
							alloc_error();
						datastore_block->next->latesttimestamp = 0;
						datastore_block->next->numentries = 0;
						datastore_block->next->data = (PIPDATA) calloc(IPDATAALLOCCHUNKS, sizeof(IPDATA));
						if(datastore_block->next->data == (PIPDATA) NULL)
							alloc_error();
						datastore_block->next->next = (PDATASTOREBLOCK) NULL;
					}
					datastore_block = datastore_block->next;
				}
			}
			return;
		} else {
			if(datastore->next == (PIPDATASTORE) NULL) {
				/* no entry for this IP, so lets make one. */
		    	        datastore->next = (PIPDATASTORE) malloc(sizeof(IPDATASTORE));
				if(datastore->next == (PIPDATASTORE) NULL)
					alloc_error();
				datastore->next->ip = ipdata->ip;
				datastore->next->next = (PIPDATASTORE) NULL;
				/* allocate its first block of storage */
				datastore->next->firstblock =
					(PDATASTOREBLOCK) malloc(sizeof(DATASTOREBLOCK));
				if(datastore->next->firstblock == (PDATASTOREBLOCK) NULL)
					alloc_error();
				datastore->next->firstblock->latesttimestamp = 0;
				datastore->next->firstblock->numentries = 0;
				datastore->next->firstblock->data =
					(PIPDATA) calloc(IPDATAALLOCCHUNKS, sizeof(IPDATA));
				if(datastore->next->firstblock->data == (PIPDATA) NULL)
					alloc_error();
				datastore->next->firstblock->next = (PDATASTOREBLOCK) NULL;
			}
			datastore = datastore->next;
		}
	}
}


/*
 * Wrapper for above, to make iteration easy.
 * Dumps data into a formatted memory area, one IP at a time.
 *
 */

static VOID store_ipdata_in_ram(IPDATA incdata[])
{	UINT i;

	for(i = 0; i < ipcount; i++)
		_store_ipdata_in_ram(&incdata[i]);
}


/*
 * Commit data from the IP table, into database, storage or graph.
 * This is done after an interval has elapsed.
 *
 */

static VOID commit_data(time_t timestamp)
{	static BOOL may_graph = TRUE;
	UINT i;
	struct stat statbuf;
	CHAR logname1[MAXPATH];		/* log.X.Y.cdf, X=tag, Y=serial */
	CHAR logname2[MAXPATH];		/* log.X.Y.cdf, X=tag, Y=serial */
	CHAR cdf_file[MAXPATH];		/* template */

	/* Set the timestamps */

	for(i = 0; i < ipcount; i++)
		iptable[i].timestamp = timestamp;

	/* Output to external database; only call this once, from
	   the parent process. */

	if(config.output_database == TRUE && config.tag == '1')
		store_ipdata_in_database(iptable);

	/* Output to CDF file. */

	if(config.output_cdf == TRUE) {
		store_ipdata_in_cdf(iptable);

		/* All of the rest here is log rotation. */

		if(rotate_logs >= config.range/RANGE1) { /* Set this++ on HUP */
			strcpy(cdf_file, config.logs);	/* path */
			strcat(cdf_file, CDF_FILE);	/* add name template */

			/* rotate 4 to 5 */
			sprintf(logname1, cdf_file, config.tag, 4);
			sprintf(logname2, cdf_file, config.tag, 5);
			if(!stat(logname2, &statbuf)) /* File exists */
				unlink(logname2);
			if(!stat(logname1, &statbuf)) /* File exists */
				rename(logname1, logname2);

			/* rotate 3 to 4 */
			sprintf(logname1, cdf_file, config.tag, 3);
			sprintf(logname2, cdf_file, config.tag, 4);
			if(!stat(logname1, &statbuf)) /* File exists */
				rename(logname1, logname2);

			/* rotate 2 to 3 */
			sprintf(logname1, cdf_file, config.tag, 2);
			sprintf(logname2, cdf_file, config.tag, 3);
			if(!stat(logname1, &statbuf)) /* File exists */
				rename(logname1, logname2);

			/* rotate 1 to 2 */
			sprintf(logname1, cdf_file, config.tag, 1);
			sprintf(logname2, cdf_file, config.tag, 2);
			if(!stat(logname1, &statbuf)) /* File exists */
				rename(logname1, logname2);

			/* rotate 0 to 1 */
			sprintf(logname1, cdf_file, config.tag, 0);
			sprintf(logname2, cdf_file, config.tag, 1);
			if(!stat(logname1, &statbuf)) /* File exists */
				rename(logname1, logname2);

			/* touch 0 to create */
			fclose(fopen(logname1, "a"));
			rotate_logs = 0;
		}
	}

	/* Generate graph. */

	if(config.graph == TRUE) {
		store_ipdata_in_ram(iptable);	/* make formatted data area */

		if(waitpid(-1, (PINT) NULL, WNOHANG))  /* A child was reaped */
			may_graph = TRUE;

		if(graphintervalcount%config.skip_intervals == 0 &&
			may_graph == TRUE) {
			may_graph = FALSE;
			/* If write_out_webpages fails, re-enable graphing
			   since there won't be any children to reap. */
			if(write_out_webpages(timestamp))
				may_graph = TRUE;
		} else {
			if(graphintervalcount%config.skip_intervals == 0)
				syslog(
					LOG_INFO,
					"(%c): previous graphing run not "
					"complete...skipping current run",
					config.tag);
		}
		drop_old_data(timestamp);
	}
}


/*
 * Return or allocate an IP's data structure.
 *
 */

static PIPDATA find_ip(UINT ipaddr)
{	UINT i;

	for(i = 0; i < ipcount; i++)
		if(iptable[i].ip == ipaddr)
			return(&iptable[i]);

	if(ipcount >= config.max_ips) {
		syslog(LOG_ERR,
			"(%c): too many IPs, dropping IP....", config.tag);
		return((PIPDATA) NULL);
        }

	memset(&iptable[ipcount], 0, sizeof(IPDATA));

	iptable[ipcount].ip = ipaddr;
	return(&iptable[ipcount++]);
}


/*
 * Convert a host IP to character form.
 *
 */

PCHAR inline hostip_to_charip(ULONG ipaddr, PCHAR buffer)
{	struct in_addr in_addr;
	PCHAR s;

	in_addr.s_addr = htonl(ipaddr);	/* convert to normalised long */
	s = inet_ntoa(in_addr);		/* convert to character form */
	strcpy(buffer, s);
	return(buffer);
}


/*
 * Fork, and orphan the child.
 *
 */

static INT fork2(VOID)
{	pid_t pid;

	if(!(pid = fork())) {
	        if(!fork()) return(0);

		exit(EXIT_SUCCESS);
	}

	waitpid(pid, NULL, 0);
	return(1);
}


/*
 * Set appropriate configuration depending on which process this is.
 *
 */

static VOID set_child_config(INT level)
{	static ULONGLONG graph_cutoff;

	config.tag = '0' + level + 1;
	config.range = range[level];
	config.interval = interval[level];
	switch(level) {
		case 0:
			graph_cutoff = config.graph_cutoff;
			break;

		case 1:	/* Overide skip_intervals for children */
			config.skip_intervals = DEFAULT_INTERVALS;
			config.graph_cutoff = graph_cutoff*(RANGE2/RANGE1);
			break;

		case 2:	/* Overide skip_intervals for children */
			config.skip_intervals = DEFAULT_INTERVALS;
			config.graph_cutoff = graph_cutoff*(RANGE3/RANGE1);
			break;

		case 3:	/* Overide skip_intervals for children */
			config.skip_intervals = DEFAULT_INTERVALS;
			config.graph_cutoff = graph_cutoff*(RANGE4/RANGE1);
			break;

		default:
			syslog(
				LOG_ERR,
				"set_child_config got an invalid level"
				" argument: %d",
				level);
			exit(EXIT_FAILURE);
	}
}


/*
 * Recover a previous CDF file and data.
 *
 */

static VOID recover_data_from_cdf(VOID)
{	FILE *cdf;
	INT i;
	BOOL first = FALSE;
	CHAR logname[MAXPATH];		/* log.X.Y.cdf, X=tag, Y=serial */
	CHAR cdf_file[MAXPATH];		/* template */

	strcpy(cdf_file, config.logs);	/* path */
	strcat(cdf_file, CDF_FILE);	/* add template */

	/* Work through the logfiles, starting at the oldest. Not
	   all of them need exist. */

	for(i = 5; i >= 0; i--) {
		sprintf(logname, cdf_file, config.tag, i);
		if(rcdf_test(logname) == TRUE)
			break;
	}

	/* 'i' now contains number of earliest useful logfile.
	   Work through this and all later files, recovering data. */

	first = TRUE;
	for(; i >= 0; i--) {
		sprintf(logname, cdf_file, config.tag, i);
		if((cdf = fopen(logname, "r"))) {
			syslog(LOG_INFO, "Recovering from %s", logname);
			if(first == TRUE) {
				rcdf_position_stream(cdf);
				first = FALSE;
			}
			rcdf_load(cdf);
		}
	}
}


/*
 * Determine if all data in the file is before the cutoff.
 * Return FALSE if file does not exist or can be ignored.
 *
 */

static BOOL rcdf_test(PCHAR filename)
{	FILE *cdf;
	CHAR ipaddrBuffer[16];
	ULONG l;
	INT status;
	time_t timestamp;

	cdf = fopen(filename, "r");
	if(cdf == (FILE *) NULL) return(FALSE);

	status = fseek(cdf, -10, SEEK_END);	/* seek to near end of file */
	while(fgetc(cdf) != '\n') {	/* rewind to last newline */
		if(fseek(cdf, -2, SEEK_CUR) == -1)
			break;
	}

	if(fscanf(
		cdf,
		" %15[0-9.],%lu,",
		ipaddrBuffer,
		&l) != 2) {
		syslog(LOG_ERR, "%s is corrupted, skipping", filename);
		return(FALSE);
	}
	timestamp = (time_t) l;	/* timestamp of last entry */
	fclose(cdf);

	if(timestamp < time(NULL) - config.range)
		/* No data in this file from after cutoff */
		return(FALSE);
	else
		/* This file has data from after cutoff */
		return(TRUE);
}


/*
 * Try to reduce unwanted data.
 *
 */

static VOID rcdf_position_stream(FILE *cdf)
{	ULONG l;
	time_t timestamp, current_timestamp;
	CHAR ipaddrBuffer[16];

	current_timestamp = time(NULL);

	fseek(cdf, 0, SEEK_END);
	timestamp = current_timestamp;
	while(timestamp > current_timestamp - config.range) {
		/* What happens if we seek past the beginning of the file? */
		if(fseek(
			cdf,
			-config.max_ips*75*(config.range/config.interval)/20,
			SEEK_CUR)) {
			/* fseek returned error, just seek to beginning */
			fseek(cdf, 0, SEEK_SET);
			return;
		}
		while(fgetc(cdf) != '\n' && !feof(cdf)); /* Read to next line */
		ungetc('\n', cdf);  /* Just so the fscanf mask stays identical */
	        if(fscanf(cdf, " %15[0-9.],%lu,", ipaddrBuffer, &l) != 2) {
			syslog(
				LOG_ERR,
				"Unknown error while scanning for beginning"
				" of data...\n");
			return;
		}
		timestamp = (time_t) l;
	}

	while(fgetc(cdf) != '\n' && !feof(cdf));
	ungetc('\n', cdf);
}


/*
 * Load data from a recovered CDF file.
 *
 */

static VOID rcdf_load(FILE *cdf)
{	time_t timestamp, current_timestamp = 0;
	struct in_addr ipaddr;
	PIPDATA ip = NULL;
	CHAR ipaddrBuffer[16];
	ULONG i, l;
	ULONG intervals_read = 0;

	for(i = 0; !feof(cdf) && !ferror(cdf); i++) {
		if(fscanf(cdf, " %15[0-9.],%lu,", ipaddrBuffer, &l) != 2)
			break;
		timestamp = (time_t) l;

		if(current_timestamp == 0) /* First run through loop */
			current_timestamp = timestamp;

		if(timestamp != current_timestamp) { /* Dump to datastore */
			store_ipdata_in_ram(iptable);
			ipcount = 0; /* reset traffic counters */
			current_timestamp = timestamp;
			intervals_read++;
		}

		inet_aton(ipaddrBuffer, &ipaddr);
		ip = find_ip(ntohl(ipaddr.s_addr));
		ip->timestamp = timestamp;

		if(fscanf(
			cdf,
			"%llu,%llu,%llu,%llu,%llu,%llu,%llu,",
			&ip->send.total,
			&ip->send.icmp,
			&ip->send.udp,
			&ip->send.tcp,
			&ip->send.ftp,
			&ip->send.http,
			&ip->send.p2p) != 7 ||
		   fscanf(
			cdf,
			"%llu,%llu,%llu,%llu,%llu,%llu,%llu",
			&ip->receive.total,
			&ip->receive.icmp,
			&ip->receive.udp,
			&ip->receive.tcp,
			&ip->receive.ftp,
			&ip->receive.http,
			&ip->receive.p2p) != 7)
				break;
	}

	store_ipdata_in_ram(iptable);
	syslog(LOG_INFO, "Finished recovering %lu records", i);
	drop_old_data(time(NULL)); /* dump the extra data */

	if(!feof(cdf))
		syslog(LOG_ERR, "Failed to parse part of log file. Giving up on the file");
	ipcount = 0;	/* Reset traffic counters */
	fclose(cdf);
}


/*
 * Signal handler. Handles HUP for logfile rotation, and TERM for cleanup.
 *
 */

static VOID signal_handler(INT sig)
{	switch(sig) {
		case SIGHUP:
			signal(SIGHUP, signal_handler);
			rotate_logs++;
			if(config.tag == '1') { /* master process */
				int i;
				/* signal children */
				for(i = 0; i < NR_WORKER_CHILDS; i++)
					kill(workerchildpids[i], SIGHUP);
			}
			break;

		case SIGTERM:
			if(config.tag == '1') {
				int i;
				/* send term signal to children */
				for(i = 0; i < NR_WORKER_CHILDS; i++)
					kill(workerchildpids[i], SIGTERM);
			}
			unlink(config.pidfile);
			exit(EXIT_SUCCESS);
	}
}


/*
 * Write out web pages containing real data.
 *
 */

static INT write_out_webpages(time_t timestamp)
{	PIPDATASTORE datastore = ipdatastore;
	PSUMMARYDATA *summary_data;
	INT numgraphs = 0;
	pid_t graphpid;
	INT i;

	/* Did we catch any packets since last time? */

	if(datastore == (PIPDATASTORE) NULL) return(-1);

	/* Run graph in child process to avoid missing packets */

	graphpid = fork();
	switch(graphpid) {
		case 0:		/* child, so run the graph */
			signal(SIGHUP, SIG_IGN); /* Not interested in HUP here */
			nice(4); /* avoid swamping CPU */
			/* count number of IPs in datastore */
			for(datastore = ipdatastore, i = 0;
				datastore != (PIPDATASTORE) NULL;
				i++, datastore = datastore->next);

			/* add 1 because we don't want to accidentally
			   allocate 0 */
			summary_data = (PSUMMARYDATA *) malloc(sizeof(PSUMMARYDATA)*i+1);
			if(summary_data == (PSUMMARYDATA *) NULL) {
				syslog(LOG_ERR,
					"(%c): graphing child failed to "
					" allocate memory", config.tag);
				exit(EXIT_FAILURE);
			}

			datastore = ipdatastore;
			while(datastore != (PIPDATASTORE) NULL) {
				if(datastore->firstblock->numentries > 0) {
					summary_data[numgraphs] =
						(PSUMMARYDATA) malloc(sizeof(SUMMARYDATA));
					if(summary_data[numgraphs] ==
						(PSUMMARYDATA) NULL) {
						syslog(LOG_ERR,
							"(%c): graphing child"
							" failed to allocate"
							" memory", config.tag);
						exit(EXIT_FAILURE);
					}
					graph_ip(
						datastore,
						summary_data[numgraphs],
						timestamp+LEAD*config.range);
					numgraphs++;
				}
				datastore = datastore->next;
			}

			make_index_pages(numgraphs, summary_data);
			exit(EXIT_SUCCESS);
			break;

		case -1:
			syslog(LOG_ERR,
				"(%c): failed to fork graphing process!",
				config.tag);
			return(-2);
			break;

		default:		/* parent */
			return(0);
			break;
	}
}


/*
 * Write out initial web pages (before there is any data to display).
 *
 */

static VOID init_webpage(PCHAR filename, CHAR tag)
{	FILE *fp;
	CHAR file[MAXPATH];

	strcpy(file, config.htdocs);	/* path */
	strcat(file, filename);		/* add actual name */

	fp = fopen(file, "w");
	if(fp == (FILE *) NULL) {
		syslog(LOG_ERR, "Cannot open %s for writing", file);
		exit(EXIT_FAILURE);
	}

	webpage_head(fp, PROGTITLE, "");
	webpage_period_links(fp, tag);

	fprintf(fp,
		"<br /><div>"PROGTITLE" has nothing to graph.\n"
		"This message should be replaced by graphs in a few minutes.\n"
		"</div>");

	webpage_tail(fp);

	fclose(fp);
}


/*
 * Output the standard webpage header and title.
 *
 */

VOID webpage_head(FILE *fp, PCHAR title, PCHAR info)
{	fprintf(fp,
		"<!DOCTYPE html"
		" PUBLIC \"-//W3C//DTD XHTML 1.0 Transitional//EN\"\n"
		"        \"http://www.w3.org/TR/xhtml1/DTD/xhtml1-transitional.dtd\">\n");

	fprintf(fp, "<html lang=\"en\">\n");

	fprintf(fp, "<head>\n");
	fprintf(fp, "<link rel=\"stylesheet\" type=\"text/css\""
			" href=\""CSS_FILE"\" />");
	fprintf(fp, "<title>%s</title>\n", title);
	if(config.meta_refresh)
		fprintf(fp,
			"<meta http-equiv=\"refresh\" content=\"%u\" />\n",
			config.meta_refresh);
	fprintf(fp, "<meta http-equiv=\"expires\" content=\"-1\" />\n");
	fprintf(fp, "<meta http-equiv=\"pragma\" content=\"no-cache\" />\n");
	fprintf(fp, "<meta http-equiv=\"content-type\" "
			"content=\"text/html; charset=UTF-8\" />\n");
	fprintf(fp, "</head>\n");

	fprintf(fp,
		"<body><a name=\"top\"></a>\n"
		"<table class=\"titlebox\" width=\"100%%\">\n"
		"<tr>\n"
		"<td class=\"smalltitle\" width=\"15%%\" "
		"align=\"center\">"
		"%s</td>\n"
		"<td class=\"title\" width=\"70%%\">"PROGTITLE"</td>\n"
		"<td class=\"smalltitle\" align=\"right\" width=\"15%%\">"
		"bandwidth<br />monitor</td>\n"
		"</tr>\n"
		"</table>\n"
		"<br />\n", info);
}


/*
 * Output the standard webpage trailer.
 *
 */

VOID webpage_tail(FILE * fp)
{	fprintf(fp, "</body>\n");
	fprintf(fp, "</html>\n");
}


/*
 * Output a brief usage message.
 *
 */

static VOID usage(VOID)
{	PCHAR *p = (PCHAR *) helpinfo;
	PCHAR q;

	for(;;) {
		q = *p++;
		if(*q == '\0') break;
		fprintf(stderr, q, PROGNAME);
		fputc('\n', stderr);
	}
	fprintf(stderr, "This is version %d.%d\n", VERSION, EDIT);
}

/*
 * End of file: bwmon.c
 *
 */
