/*
 * File: graph.c
 *
 */

#include <unistd.h>
#include <setjmp.h>
#include <signal.h>
#include <string.h>
#include <netdb.h>
#include <netinet/in.h>

#include <sys/types.h>
#include <sys/socket.h>

#include <time.h>
#include "bwmon.h"

#include <arpa/nameser.h>
#include <resolv.h>

/* Forward references */

static	ULONGLONG	graph_data(gdImagePtr, gdImagePtr, PIPDATASTORE,
				time_t, PSUMMARYDATA);
static	VOID		prepare_x_axis(gdImagePtr, time_t);
static	VOID		prepare_y_axis(gdImagePtr, ULONGLONG);
static	VOID		rdnslngjmp(INT signal);
static	VOID		table_heading(FILE *);
static	VOID		webpage_graphs(FILE *, CHAR [], CHAR []);

/* Local storage */

static	jmp_buf		dnsjump;


/*
 * Reverse DNS lookup, with timeout.
 *
 */

#define	MAX_DNS_TIMEOUTS	100

static VOID rdns(PCHAR buffer, ULONG IP)
{	static const char too_many_dns_timeouts[] =
		"Too many DNS timeouts, reverse lookups suspended";
	struct hostent *hostent;
	CHAR chr_ip[50];
	ULONG addr = htonl(IP);
	static BOOL initialised = FALSE;
	static INT dns_timeouts = 0;  /* reset each time (independent processes) */

	if(initialised == FALSE) {	/* prime alarm handler */
		signal(SIGALRM, rdnslngjmp);
		_res.retrans = 1;	/* set resolver values */
		_res.retry = 2;
		res_init();

		initialised = TRUE;
	}

	if(dns_timeouts > MAX_DNS_TIMEOUTS) {
		syslog(LOG_ERR, too_many_dns_timeouts);
		strncpy(buffer, too_many_dns_timeouts, MAX_HOSTNAME-2);
		buffer[MAX_HOSTNAME-1] = '\0';
		return;
	}

	if(setjmp(dnsjump) == 0) {	/* return here on timeout, nonzero */
		alarm(DNS_TIMEOUT);	/* Don't let lookup stall us */
		hostent = gethostbyaddr((PCHAR) &addr, 4, AF_INET);
					/* (PCHAR)&data->IP */
		alarm(0);		/* cancel pending alarm */

		if(hostent != (struct hostent *) NULL)
			sprintf(buffer, "%s", hostent->h_name);
		else {
			strncpy(
				buffer,
				"Configure DNS to reverse this IP",
				MAX_HOSTNAME-2);
			buffer[MAX_HOSTNAME-1] = '\0';
		}
	} else {			/* timed out */
		hostip_to_charip(IP, chr_ip);
		syslog(LOG_ERR,
			"DNS timeout for %s: "
			"this problem reduces graphing performance", chr_ip);
		dns_timeouts++;
		strncpy(
			buffer,
			"DNS timeout: correct to speed up graphing",
			MAX_HOSTNAME-2);
		buffer[MAX_HOSTNAME-1] = '\0';
	}
}


/*
 * DNS timeout handler. Simply longjmps back into 'rdns'.
 *
 */

static VOID rdnslngjmp(INT signal)
{	longjmp(dnsjump, 1);
}


/*
 * Swap summary data; used by sort.
 *
 */

static VOID swap(PSUMMARYDATA *a, PSUMMARYDATA *b)
{	PSUMMARYDATA temp;

	temp = *a;
	*a = *b;
	*b = temp;
}


/*
 * Sort the summary data by the total field.
 *
 */

static VOID quicksort_summary_data(PSUMMARYDATA summary_data[],
	INT left, INT right)
{	INT i,j,centre;
	ULONGLONG pivot;

	if(left == right) return;	/* just one item left */

	if(left+1 == right) {		/* just two items left */
		if(summary_data[left]->total < summary_data[right]->total)
			swap(&summary_data[left],&summary_data[right]);
		return;
	}

	/* use the median-of-three method for picking pivot */

	centre = (left+right)/2;
	if(summary_data[left]->total < summary_data[centre]->total)
		swap(&summary_data[left],&summary_data[centre]);
	if(summary_data[left]->total < summary_data[right]->total)
		swap(&summary_data[left],&summary_data[right]);
	if(summary_data[centre]->total < summary_data[right]->total)
		swap(&summary_data[centre],&summary_data[right]);
	pivot = summary_data[centre]->total;
	swap(&summary_data[centre],&summary_data[right-1]); /* hide the pivot */

	i = left; j = right - 1;

	do {
		do { ++i; } while(summary_data[i]->total > pivot);
		do { --j; } while(summary_data[j]->total < pivot);
		swap(&summary_data[i],&summary_data[j]);
	} while(j > i);

	swap(&summary_data[i],&summary_data[j]); /* undo last swap */
	swap(&summary_data[i],&summary_data[right-1]); /* restore pivot */
	quicksort_summary_data(summary_data,left,i-1);
	quicksort_summary_data(summary_data,i+1,right);
}


/*
 * Format a numeric table element and send it to the output file.
 *
 */

#define NUM_FACTOR 1024

static VOID format_num(ULONGLONG n, PCHAR buf, INT len)
{	DOUBLE f;

	if(n < NUM_FACTOR) {
		snprintf(buf, len, "<td><tt>%i&nbsp;</tt></td>",
			(INT) n);
		return;
	}

	f = n;
	f /= NUM_FACTOR;
	if(f < NUM_FACTOR) {
		snprintf(buf, len, "<td><tt>%.1fk</tt></td>",f);
		return;
	}

	f /= NUM_FACTOR;
	if(f < NUM_FACTOR) {
		snprintf(buf, len, "<td><tt>%.1fM</tt></td>",f);
		return;
	}

	f /= NUM_FACTOR;
	if(f < NUM_FACTOR) {
		snprintf(buf, len, "<td><tt>%.1fG</tt></td>",f);
		return;
	}

	f /= NUM_FACTOR;
	snprintf(buf, len, "<td><tt>%.1fT</tt></td>\n",f);
}


/*
 * Print a line in a web page table.
 *
 */

#define	MAX_NUM		50

static VOID print_table_line(FILE *fp, PSUMMARYDATA data, INT counter)
{	CHAR buffer1[MAX_NUM];
	CHAR buffer2[MAX_NUM];
	CHAR buffer3[MAX_NUM];
	CHAR buffer4[MAX_NUM];
	CHAR buffer4b[MAX_NUM];
	CHAR buffer5[MAX_NUM];
	CHAR buffer5b[MAX_NUM];
	CHAR buffer6[MAX_NUM];
	CHAR buffer7[MAX_NUM];
	CHAR buffer8[MAX_NUM];

	/* First convert the information to human readable form */

	if(data->ip == 0)
		strcpy(buffer1, "Total");
	else
		hostip_to_charip(data->ip, buffer1);

	format_num(data->total,		buffer2,  MAX_NUM);
	format_num(data->totalsent,	buffer3,  MAX_NUM);
	format_num(data->totalreceived,	buffer4,  MAX_NUM);
	format_num(data->ftp,		buffer4b, MAX_NUM);
	format_num(data->http,		buffer5,  MAX_NUM);
	format_num(data->p2p,		buffer5b, MAX_NUM);
	format_num(data->tcp,		buffer6,  MAX_NUM);
	format_num(data->udp,		buffer7,  MAX_NUM);
	format_num(data->icmp,		buffer8,  MAX_NUM);

	if(counter%4 == 0 || (counter-1)%4 == 0)
		fprintf(fp, "<tr align=\"right\">");
	else
		fprintf(fp, "<tr align=\"right\" class=\"highlight\">");

	if(data->graph == TRUE)
		fprintf(fp,
			"<td align=\"left\"><a href=\"#%s-%c\">%s</a></td>",
			buffer1,	/* IP */
			config.tag,
			buffer1);	/* IP */
	else
		fprintf(fp,
			"<td align=\"left\">%s</td>",
			buffer1);	/* IP */

	fprintf(fp,
		"%s%s%s%s%s%s%s%s%s</tr>\n",
		buffer2,	/* Total */
		buffer3,	/* Total sent */
		buffer4,	/* Total rcvd */
		buffer4b,	/* FTP */
		buffer5,	/* HTTP */
		buffer5b,	/* P2P */
		buffer6,	/* TCP */
		buffer7,	/* UDP */
		buffer8);	/* ICMP */
}


/*
 * Build the web pages, using the graphs and the summary data.
 * Also frees the summary data area.
 *
 */

VOID make_index_pages(INT num_ips, PSUMMARYDATA summary_data[])
{	INT snc;
	INT i, j;
	time_t write_time;
	PCHAR file;
	PCHAR period_desc;
	FILE *fp;
	CHAR buffer1[50];
	CHAR buffer2[50];
	CHAR buffer3[50];
	CHAR hostname[MAX_HOSTNAME];
	CHAR filename[MAXPATH];

	write_time = time(NULL);

	quicksort_summary_data(summary_data, 0, num_ips - 1);

	/* Print main index page */

	switch(config.tag) {
		case '1':	file = INDEX_HTML;
				period_desc = "Daily"; break;
		case '2':	file = INDEX2_HTML;
				period_desc = "Weekly"; break;
		case '3':	file = INDEX3_HTML;
				period_desc = "Monthly"; break;
		case '4':	file = INDEX4_HTML;
				period_desc = "Yearly"; break;
		default:	syslog(LOG_ERR,
					"bad config tag (%c) in "
					"make_index_pages", config.tag);
				exit(EXIT_FAILURE);
	}

	strcpy(filename, config.htdocs);	/* path */
	strcat(filename, file);			/* actual name */
	fp = fopen(filename, "w");
	if(fp == (FILE *) NULL) {
		syslog(LOG_ERR, "Failed to open %s", filename);
		exit(EXIT_FAILURE);
	}

	webpage_head(fp, PROGTITLE, ctime(&write_time));
	webpage_period_links(fp, config.tag);

	fprintf(fp,
		"<br />\n&nbsp;&nbsp;&nbsp;Top%d&nbsp;&nbsp;",
		config.top);

	for(i = 0; i < subnet_count; i++) {
		hostip_to_charip(subnet_table[i].ip, buffer1);
		fprintf(fp,
			"&nbsp;&nbsp;<a href=\"subnet-%c-%s.html\">%s</a>"
			"&nbsp;&nbsp;",
			config.tag, buffer1, buffer1);
	}

	/* Top 'top' */

	fprintf(fp, "<h1>Top %d IPs by traffic - %s</h1>\n",
			config.top, period_desc);

	/* Pass 1: write out the table */

	table_heading(fp);
	for(i = 0; i <= config.top && i < num_ips; i++)
		print_table_line(fp, summary_data[i], i);

	fprintf(fp, "</table>\n</div>\n");

	/* Pass 2: the graphs */

	for(i = 0; i <= config.top && i < num_ips; i++) {
		if(summary_data[i]->graph == TRUE) {
			if(summary_data[i]->ip == 0) {
				strcpy(buffer1, "Total");
				strcpy(hostname, "Total of all subnets");
			} else {
				hostip_to_charip(summary_data[i]->ip, buffer1);
				rdns(hostname, summary_data[i]->ip);
			}
			webpage_graphs(fp, buffer1, hostname);
		}
	}
	webpage_tail(fp);
	fclose(fp);

	/* Print each subnet page */

	for(snc = 0; snc < subnet_count; snc++) {
		INT current_ip = subnet_table[snc].ip;

		hostip_to_charip(current_ip, buffer1);
		sprintf(buffer2, SUBNET_HTML, config.tag, buffer1);
		strcpy(filename, config.htdocs);	/* path */
		strcat(filename, buffer2);		/* as above */
		fp = fopen(filename, "w");

		sprintf(buffer3, PROGTITLE" - subnet %s", buffer1);
		webpage_head(fp, buffer3, ctime(&write_time));
		webpage_period_links(fp, config.tag);

		if(config.tag == '1')
			fprintf(fp,
				"<br />\n&nbsp;&nbsp;&nbsp;"
				"<a href=\""INDEX_HTML"\">Top%d</a>"
				"&nbsp;&nbsp;", config.top);
		else
			fprintf(fp,
				"<br />\n&nbsp;&nbsp;&nbsp;"
				"<a href=\""INDEXT_HTML"\">Top%d</a>"
				"&nbsp;&nbsp;",
				config.tag, config.top);

		for(i = 0; i < subnet_count; i++) {
			hostip_to_charip(subnet_table[i].ip, buffer2);
			if(current_ip == subnet_table[i].ip) {
				fprintf(fp,
					"&nbsp;&nbsp;%s&nbsp;&nbsp;",
					buffer2);
			} else {
				fprintf(fp,
					"&nbsp;&nbsp;"
					"<a href=\"subnet-%c-%s.html\">%s</a>"
					"&nbsp;&nbsp;",
					config.tag, buffer2, buffer2);
			}
		}

		fprintf(fp, "<h1>%s - %s</h1>", buffer1, period_desc);

		/* Pass 1: write out the table */

		table_heading(fp);
		for(j = 0, i = 0; i < num_ips; i++) {
			 if(current_ip ==
				(summary_data[i]->ip & subnet_table[snc].mask)) {
				/* The IP belongs to this subnet */
				print_table_line(fp, summary_data[i], j++);
			}
		}

		fprintf(fp, "</table>\n</div>\n");

		/* Pass 2: the graphs */

		for(i = 0; i < num_ips; i++) {
			if(current_ip ==
				(summary_data[i]->ip & subnet_table[snc].mask)) {
				/* The IP belongs to this subnet */
				if(summary_data[i]->graph) {
					hostip_to_charip(
						summary_data[i]->ip, buffer1);
					rdns(hostname, summary_data[i]->ip);
					webpage_graphs(fp, buffer1, hostname);
				}
			}
		}
		webpage_tail(fp);
		fclose(fp);
	}
	free(summary_data);
}


/*
 * Selectively output the period links.
 *
 */

VOID webpage_period_links(FILE * fp, CHAR tag)
{	fprintf(fp, "&nbsp;&nbsp;&nbsp;");

	fprintf(fp,
		tag == '1' ?
		"Daily&nbsp;&nbsp;&nbsp;&nbsp;" :
		"<a href=\""INDEX_HTML"\">Daily</a>&nbsp;&nbsp;&nbsp;&nbsp;");

	fprintf(fp,
		tag == '2' ?
		"Weekly&nbsp;&nbsp;&nbsp;&nbsp;" :
		"<a href=\""INDEX2_HTML"\">Weekly</a>&nbsp;&nbsp;&nbsp;&nbsp;");

	fprintf(fp,
		tag == '3' ?
		"Monthly&nbsp;&nbsp;&nbsp;&nbsp;" :
		"<a href=\""INDEX3_HTML"\">Monthly</a>&nbsp;&nbsp;&nbsp;&nbsp;");

	fprintf(fp,
		tag == '4' ?
		"Yearly" :
		"<a href=\""INDEX4_HTML"\">Yearly</a>");

	fprintf(fp, "<br />\n");
}


/*
 * Output a pair of graphs.
 *
 */

static VOID webpage_graphs(FILE *fp, CHAR buf[], CHAR hostname[])
{	fprintf(fp,
		"<a name=\"%s-%c\"></a>"
		"<h1><a href=\"#top\">(Top)</a>"
		" %s - %s</h1><br />\n"
		"Sent:<br />\n"
		"<img src=\"%s-%c-S.png\" "
		"alt=\"Sent traffic for %s\" /><br />\n"
		"<img src=\""LEGEND_FILE"\" "
		"alt=\"Legend\" /><br />\n"
		"Received:<br />\n"
		"<img src=\"%s-%c-R.png\" "
		"alt=\"Sent traffic for %s\" /><br />\n"
		"<img src=\""LEGEND_FILE"\" "
		"alt=\"Legend\" /><br />\n"
		"<br />\n",
		buf, config.tag, buf, hostname,
		buf, config.tag, buf, buf,
		config.tag, buf);
}


/*
 * Output a table heading.
 *
 */

static VOID table_heading(FILE * fp)
{	fprintf(fp,
		"<div align=\"center\">\n"
		"<table class=\"stats\" width=\"95%%\""
		" cellspacing=\"0\">\n"
		"<tr class=\"heading\">"
		"<th width=\"11%%\">IP address</th>"
		"<th width=\"9%%\">Total</th>"
		"<th width=\"9%%\">Total sent</th>"
		"<th width=\"9%%\">Total rcvd</th>"
		"<th width=\"9%%\">FTP</th>"
		"<th width=\"9%%\">HTTP</th>"
		"<th width=\"9%%\">P2P</th>"
		"<th width=\"9%%\">TCP</th>"
		"<th width=\"9%%\">UDP</th>"
		"<th width=\"9%%\">ICMP</th>"
		"</tr>\n");
}


/*
 * Generate graph (two files) for a single IP address.
 * Summary data is returned via the parameter.
 *
 */

VOID graph_ip(PIPDATASTORE datastore, PSUMMARYDATA summary_data,
		time_t timestamp)
{	FILE *ofp;
	CHAR sfile[MAXPATH], rfile[MAXPATH], temp[MAXPATH];
	gdImagePtr im, im2;
	INT white;
	ULONGLONG ymax;
	CHAR charip[20];		/* IP number or "Total" tag */
	time_t graph_begin_time;

	/* Set title */

	if(datastore->ip == 0)
		strcpy(charip, "Total");
	else
		hostip_to_charip(datastore->ip, charip);

	graph_begin_time = timestamp - config.range;

	/* Create two blank white images */

	im = gdImageCreate(XWIDTH, YHEIGHT);
	white = gdImageColorAllocate(im, 255, 255, 255);
	im2 = gdImageCreate(XWIDTH, YHEIGHT);
	white = gdImageColorAllocate(im2, 255, 255, 255);

	/* Generate filenames for the graphs */

	sprintf(temp, S_GRAPH, charip, config.tag);
	strcpy(sfile, config.htdocs);	/* path */
	strcat(sfile, temp);

	sprintf(temp, R_GRAPH, charip, config.tag);
	strcpy(rfile, config.htdocs);	/* path */
	strcat(rfile, temp);

	/* Build the graphs */

	ymax = graph_data(im, im2, datastore, graph_begin_time, summary_data);
	if(ymax != 0) {	/* Finish the graph if there is anything there */
		prepare_x_axis(im, timestamp);
		prepare_y_axis(im, ymax);
		ofp = fopen(sfile, "w");
		gdImagePng(im, ofp);
		fclose(ofp);

		prepare_x_axis(im2, timestamp);
		prepare_y_axis(im2, ymax);
		ofp = fopen(rfile, "w");
		gdImagePng(im2, ofp);
		fclose(ofp);
	} else {  /* the graph isn't worth cluttering up the web pages with */
		unlink(sfile);		/* delete out of date graph files */
		unlink(rfile);
	}

	gdImageDestroy(im);
	gdImageDestroy(im2);
}


/*
 * Graph the data from 'datastore', and return summary data via 'summary_data'.
 * Graph images returned via 'im1' and 'im2'.
 * Returns ymax for the graph.
 *
 */

#define	MAX_PEAK	100
#define	MAX_DATA	100

static ULONGLONG graph_data(gdImagePtr im, gdImagePtr im2,
			PIPDATASTORE datastore, time_t timestamp,
			PSUMMARYDATA summary_data)
{	ULONGLONG ymax = 0;
	PDATASTOREBLOCK current_block;
	PIPDATA data;
	size_t datapoints;
	DOUBLE x;
	INT xint, i;
	CHAR buffer[MAX_DATA];
	CHAR buffer2[MAX_PEAK];
	INT blue, lblue, red, yellow, purple, green, brown, black;
	INT blue2, lblue2, red2, yellow2, purple2, green2, brown2, black2;
	ULONGLONG sent_peak = 0, received_peak = 0;

	ULONGLONG total[XWIDTH];
	ULONGLONG icmp[XWIDTH];
	ULONGLONG udp[XWIDTH];
	ULONGLONG tcp[XWIDTH];
	ULONGLONG ftp[XWIDTH];
	ULONGLONG http[XWIDTH];
	ULONGLONG p2p[XWIDTH];
	INT count[XWIDTH];
	ULONGLONG total2[XWIDTH];
	ULONGLONG icmp2[XWIDTH];
	ULONGLONG udp2[XWIDTH];
	ULONGLONG tcp2[XWIDTH];
	ULONGLONG ftp2[XWIDTH];
	ULONGLONG http2[XWIDTH];
	ULONGLONG p2p2[XWIDTH];

	/* Initialise counters */

	memset(count, 0, sizeof(count[0])*XWIDTH);
	memset(total, 0, sizeof(total[0])*XWIDTH);
	memset(icmp,  0, sizeof(total[0])*XWIDTH);
	memset(udp,   0, sizeof(total[0])*XWIDTH);
	memset(tcp,   0, sizeof(total[0])*XWIDTH);
	memset(ftp,   0, sizeof(total[0])*XWIDTH);
	memset(http,  0, sizeof(total[0])*XWIDTH);
	memset(p2p,   0, sizeof(total[0])*XWIDTH);

	memset(total2,0, sizeof(total[0])*XWIDTH);
	memset(icmp2, 0, sizeof(total[0])*XWIDTH);
	memset(udp2,  0, sizeof(total[0])*XWIDTH);
	memset(tcp2,  0, sizeof(total[0])*XWIDTH);
	memset(ftp2,  0, sizeof(total[0])*XWIDTH);
	memset(http2, 0, sizeof(total[0])*XWIDTH);
	memset(p2p2,  0, sizeof(total[0])*XWIDTH);

	/* Initialise output area */

	memset(summary_data, 0, sizeof(SUMMARYDATA));

	/* Set up colours for both images */

	yellow   = gdImageColorAllocate(im, 255, 255, 0);
	purple   = gdImageColorAllocate(im, 255, 0, 255);
	green    = gdImageColorAllocate(im, 0, 255, 0);
	blue     = gdImageColorAllocate(im, 0, 0, 255);
	lblue	 = gdImageColorAllocate(im, 173, 216, 230);
	brown    = gdImageColorAllocate(im, 128, 0, 0);
	red      = gdImageColorAllocate(im, 255, 0, 0);
	black 	 = gdImageColorAllocate(im, 0, 0, 0);

	yellow2  = gdImageColorAllocate(im2, 255, 255, 0);
	purple2  = gdImageColorAllocate(im2, 255, 0, 255);
	green2   = gdImageColorAllocate(im2, 0, 255, 0);
	blue2    = gdImageColorAllocate(im2, 0, 0, 255);
	lblue2	 = gdImageColorAllocate(im2, 173, 216, 230);
	brown2   = gdImageColorAllocate(im2, 128, 0, 0);
	red2     = gdImageColorAllocate(im2, 255, 0, 0);
	black2   = gdImageColorAllocate(im2, 0, 0, 0);

	current_block = datastore->firstblock;
	data = current_block->data;
	datapoints = current_block->numentries;
	summary_data->ip = data[0].ip;

	/* Sum up the bytes/second */

	while(datapoints > 0) {		/* we have data to graph */
		for(i = 0; i < datapoints; i++) {	/* graph it all */
			x = (data[i].timestamp-timestamp)*
				((XWIDTH-XOFFSET)/config.range)+XOFFSET;
			xint = x;

			if(xint >= 0 && xint < XWIDTH) {
				count[xint]++;
				if(data[i].send.total > sent_peak)
					sent_peak = data[i].send.total;
				total[xint]  += data[i].send.total;
				icmp[xint]   += data[i].send.icmp;
				udp[xint]    += data[i].send.udp;
				tcp[xint]    += data[i].send.tcp;
				ftp[xint]    += data[i].send.ftp;
				http[xint]   += data[i].send.http;
				p2p[xint]    += data[i].send.p2p;

				if(data[i].receive.total > received_peak)
					received_peak = data[i].receive.total;
				total2[xint] += data[i].receive.total;
				icmp2[xint]  += data[i].receive.icmp;
				udp2[xint]   += data[i].receive.udp;
				tcp2[xint]   += data[i].receive.tcp;
				ftp2[xint]   += data[i].receive.ftp;
				http2[xint]  += data[i].receive.http;
				p2p2[xint]   += data[i].receive.p2p;
			}
		} /* for loop */
		/* Move to next block and reset values */
		current_block = current_block->next;
		if(current_block != (PDATASTOREBLOCK) NULL) {
			data = current_block->data;
			datapoints = current_block->numentries;
		} else
			datapoints = 0;
	} /* while datapoints > 0 */

	/* Convert sent_peak and received_peak from bytes to bytes/second */

	sent_peak /= config.interval;
	received_peak /= config.interval;

	/* Perform the average */

	for(i = XOFFSET + 1; i < XWIDTH; i++) {
		if(count[i] > 0) {
			summary_data->total         += (total[i] + total2[i]);
			summary_data->totalsent     += total[i];
 			summary_data->totalreceived += total2[i];
			summary_data->tcp           += (tcp[i]  + tcp2[i]);
			summary_data->ftp           += (ftp[i]  + ftp2[i]);
			summary_data->http          += (http[i] + http2[i]);
			summary_data->p2p           += (p2p[i]  + p2p2[i]);
			summary_data->udp           += (udp[i]  + udp2[i]);
			summary_data->icmp          += (icmp[i] + icmp2[i]);

			total[i]  /= (count[i]*config.interval);
			tcp[i]    /= (count[i]*config.interval);
			ftp[i]    /= (count[i]*config.interval);
			http[i]   /= (count[i]*config.interval);
			p2p[i]    /= (count[i]*config.interval);
			udp[i]    /= (count[i]*config.interval);
			icmp[i]   /= (count[i]*config.interval);

			total2[i] /= (count[i]*config.interval);
			tcp2[i]   /= (count[i]*config.interval);
			ftp2[i]   /= (count[i]*config.interval);
			http2[i]  /= (count[i]*config.interval);
			p2p2[i]   /= (count[i]*config.interval);
			udp2[i]   /= (count[i]*config.interval);
			icmp2[i]  /= (count[i]*config.interval);

			if(total[i] > ymax) ymax = total[i];
			if(total2[i] > ymax) ymax = total2[i];
		}
	}

	ymax += ymax*0.05;		/* add an extra 5% */

	if((summary_data->ip != 0 &&
	    summary_data->total < config.graph_cutoff)) {
		/* not enough to be wanted */
		summary_data->graph = FALSE;
		return(0);
	}

	summary_data->graph = TRUE;

	/* Plot the points */

	for(i = XOFFSET+1; i < XWIDTH; i++) {
		if(count[i] > 0) {
			/* Convert the bytes/sec to y coords */
			total[i]  = (total[i]*(YHEIGHT-YOFFSET))/ymax;
			tcp[i]    = (tcp[i]*(YHEIGHT-YOFFSET))/ymax;
			ftp[i]    = (ftp[i]*(YHEIGHT-YOFFSET))/ymax;
			http[i]   = (http[i]*(YHEIGHT-YOFFSET))/ymax;
			p2p[i]    = (p2p[i]*(YHEIGHT-YOFFSET))/ymax;
			udp[i]    = (udp[i]*(YHEIGHT-YOFFSET))/ymax;
			icmp[i]   = (icmp[i]*(YHEIGHT-YOFFSET))/ymax;

			total2[i] = (total2[i]*(YHEIGHT-YOFFSET))/ymax;
			tcp2[i]   = (tcp2[i]*(YHEIGHT-YOFFSET))/ymax;
			ftp2[i]   = (ftp2[i]*(YHEIGHT-YOFFSET))/ymax;
			http2[i]  = (http2[i]*(YHEIGHT-YOFFSET))/ymax;
			p2p2[i]   = (p2p2[i]*(YHEIGHT-YOFFSET))/ymax;
			udp2[i]   = (udp2[i]*(YHEIGHT-YOFFSET))/ymax;
			icmp2[i]  = (icmp2[i]*(YHEIGHT-YOFFSET))/ymax;

			/* Stack the consolidated displays.
			   Total is stacked from the bottom
			   ICMP is on the bottom too
			   UDP is stacked on top of ICMP */

			udp[i]   += icmp[i];
			udp2[i]  += icmp2[i];
			/* TCP and P2P are stacked on top of UDP */
			tcp[i]   += udp[i];
			tcp2[i]  += udp2[i];
			p2p[i]   += udp[i];
			p2p2[i]  += udp2[i];
			/* HTTP is stacked on top of P2P */
			http[i]  += p2p[i];
			http2[i] += p2p2[i];
			/* FTP is stacked on top of HTTP */
			ftp[i]   += http[i];
			ftp2[i]  += http2[i];

			/* Now plot it all */

			/*  Sent data */

			gdImageLine(im, i, (YHEIGHT-YOFFSET) - total[i], i, YHEIGHT-YOFFSET-1, yellow);
			gdImageLine(im, i, (YHEIGHT-YOFFSET) - icmp[i], i, YHEIGHT-YOFFSET-1, red);
			gdImageLine(im, i, (YHEIGHT-YOFFSET) - udp[i], i, (YHEIGHT-YOFFSET) - icmp[i] - 1, brown);
			gdImageLine(im, i, (YHEIGHT-YOFFSET) - tcp[i], i, (YHEIGHT-YOFFSET) - udp[i] - 1, green);
			gdImageLine(im, i, (YHEIGHT-YOFFSET) - p2p[i], i, (YHEIGHT-YOFFSET) - udp[i] - 1, purple);
			gdImageLine(im, i, (YHEIGHT-YOFFSET) - http[i], i, (YHEIGHT-YOFFSET) - p2p[i] - 1, blue);
			gdImageLine(im, i, (YHEIGHT-YOFFSET) - ftp[i], i, (YHEIGHT-YOFFSET) - http[i] - 1, lblue);

			/* Received */

			gdImageLine(im2, i, (YHEIGHT-YOFFSET) - total2[i], i, YHEIGHT-YOFFSET-1, yellow2);
			gdImageLine(im2, i, (YHEIGHT-YOFFSET) - icmp2[i], i, YHEIGHT-YOFFSET-1, red2);
			gdImageLine(im2, i, (YHEIGHT-YOFFSET) - udp2[i], i, (YHEIGHT-YOFFSET) - icmp2[i] - 1, brown2);
			gdImageLine(im2, i, (YHEIGHT-YOFFSET) - tcp2[i], i, (YHEIGHT-YOFFSET) - udp2[i] - 1, green2);
			gdImageLine(im2, i, (YHEIGHT-YOFFSET) - p2p2[i], i, (YHEIGHT-YOFFSET) - udp2[i] - 1, purple2);
			gdImageLine(im2, i, (YHEIGHT-YOFFSET) - http2[i], i, (YHEIGHT-YOFFSET) - p2p2[i] - 1, blue2);
			gdImageLine(im2, i, (YHEIGHT-YOFFSET) - ftp2[i], i, (YHEIGHT-YOFFSET) - http2[i] - 1, lblue2);
		}
	}

	/* Generate annotation for the graphs */

	if(sent_peak < 1024/8)
		snprintf(buffer2, MAX_PEAK, "Peak Send Rate: %.1f bits/sec",
			(DOUBLE)sent_peak*8);
	else if(sent_peak < (1024*1024)/8)
		snprintf(buffer2, MAX_PEAK, "Peak Send Rate: %.1f kbits/sec",
			((DOUBLE)sent_peak*8.0)/1024.0);
	else
		snprintf(buffer2, MAX_PEAK, "Peak Send Rate: %.1f Mbits/sec",
		((DOUBLE)sent_peak*8.0)/(1024.0*1024.0));

	if(summary_data->totalsent < 1024)
		snprintf(buffer, MAX_DATA, "Sent %.1f Bytes",
			(DOUBLE)summary_data->totalsent);
	else if(summary_data->totalsent < 1024*1024)
		snprintf(buffer, MAX_DATA, "Sent %.1f kBytes",
			(DOUBLE)summary_data->totalsent/1024.0);
	else
		snprintf(buffer, MAX_DATA, "Sent %.1f MBytes",
		(DOUBLE)summary_data->totalsent/(1024.0*1024.0));

	gdImageString(im, gdFontSmall, XOFFSET+5,  YHEIGHT-20,
		(PUCHAR) buffer, black);
	gdImageString(im, gdFontSmall, XWIDTH/2+XOFFSET/2,  YHEIGHT-20,
		(PUCHAR) buffer2, black);

	if(received_peak < 1024/8)
	   	snprintf(buffer2, MAX_PEAK, "Peak Receive Rate: %.1f bits/sec",
			(DOUBLE)received_peak*8);
	else if(received_peak < (1024*1024)/8)
		snprintf(buffer2, MAX_PEAK, "Peak Receive Rate: %.1f kbits/sec",
			((DOUBLE)received_peak*8.0)/1024.0);
	else
		snprintf(buffer2, MAX_PEAK, "Peak Receive Rate: %.1f Mbits/sec",
			((DOUBLE)received_peak*8.0)/(1024.0*1024.0));

	if(summary_data->totalreceived < 1024)
		snprintf(buffer, MAX_DATA, "Received %.1f Bytes",
			(DOUBLE)summary_data->totalreceived);
	else if(summary_data->totalreceived < 1024*1024)
		snprintf(buffer, MAX_DATA, "Received %.1f kBytes",
			(DOUBLE)summary_data->totalreceived/1024.0);
	else
		snprintf(buffer, MAX_DATA, "Received %.1f MBytes",
			(DOUBLE)summary_data->totalreceived/(1024.0*1024.0));

	gdImageString(im2, gdFontSmall, XOFFSET+5,  YHEIGHT-20,
		(PUCHAR) buffer, black2);
	gdImageString(im2, gdFontSmall, XWIDTH/2+XOFFSET/2,  YHEIGHT-20,
		(PUCHAR) buffer2, black2);

	return(ymax);
}


/*
 * Set up the Y axis for a graph.
 *
 */

#define	Y_LEGEND	20

static VOID prepare_y_axis(gdImagePtr im, ULONGLONG ymax)
{	CHAR buffer[Y_LEGEND];
	CHAR ylegend;
	LONGLONG divisor;
	INT black;
	FLOAT ytick = 0;
	DOUBLE y;
	LONG ystep;

	black = gdImageColorAllocate(im, 0, 0, 0);
	gdImageLine(im, XOFFSET, 0, XOFFSET, YHEIGHT, black);

	ylegend = ' ';
	divisor = 1;
	if(ymax*8 > 1024*2) {
		divisor = 1024;    // Display in K
		ylegend = 'k';
	}
	if(ymax*8 > 1024*1024*2) {
		divisor = 1024*1024; // Display in M
		ylegend = 'M';
	}
	if(ymax*8 > (long long)1024*1024*1024*2) {
		divisor = 1024*1024*1024; // Display in G
		ylegend = 'G';
	}

	ystep = ymax/10;
	if(ystep < 1) ystep = 1;
	ytick = ystep;

	while(ytick < (ymax - ymax/10)) {
		y = (YHEIGHT-YOFFSET)-((ytick*(YHEIGHT-YOFFSET))/ymax);
		gdImageLine(im, XOFFSET, y, XWIDTH, y, black);
		snprintf(buffer, Y_LEGEND, "%4.1f %cbits/s",
			(FLOAT)(8.0*ytick)/divisor, ylegend);
		gdImageString(im, gdFontSmall, 3, y-7, (PUCHAR) buffer, black);
		ytick += ystep;
	}
}


/*
 * Set up the X axis for a graph.
 *
 */

#define	X_LEGEND	100

static VOID prepare_x_axis(gdImagePtr im, time_t timestamp)
{	CHAR buffer[X_LEGEND];
	INT black, grey;
	time_t sample_begin, sample_end, mt;
	struct tm *timestruct;
	LONG marktime, marktimestep;
	DOUBLE x;

	sample_begin = timestamp - config.range;
	sample_end = sample_begin + config.interval;

	black = gdImageColorAllocate(im, 0, 0, 0);
	grey = gdImageColorAllocate(im, 190, 190, 190);

	gdImageLine(im, 0, YHEIGHT-YOFFSET, XWIDTH, YHEIGHT-YOFFSET, black);

	/* Write the day/month separator bars */

	if((24*60*60*(XWIDTH-XOFFSET))/config.range > (XWIDTH-XOFFSET)/10) {
		/* Day bars */
		timestruct = localtime((time_t *)&sample_begin);
		timestruct->tm_sec = 0;
		timestruct->tm_min = 0;
		timestruct->tm_hour = 0;
		marktime = mktime(timestruct);

		x = (marktime-sample_begin)*(((DOUBLE)(XWIDTH-XOFFSET)) / config.range) + XOFFSET;
		while(x < XOFFSET) {
			marktime += (24*60*60);
			x = (marktime-sample_begin)*((XWIDTH-XOFFSET)/config.range) + XOFFSET;
		}

		while(x < (XWIDTH-10)) {
			/* Day Lines */
			gdImageLine(im, x, 0, x, YHEIGHT-YOFFSET, grey);
			gdImageLine(im, x+1, 0, x+1, YHEIGHT-YOFFSET, grey);

			mt = (time_t) marktime;
			timestruct = localtime(&mt);
			strftime(buffer, X_LEGEND, "%a, %b %d", timestruct);
			gdImageString(im, gdFontSmall, x-30,
				YHEIGHT-YOFFSET+10, (PUCHAR) buffer, black);

			/* calculate next x */

			marktime += (24*60*60);
			x = (marktime-sample_begin)*((XWIDTH-XOFFSET)/config.range) + XOFFSET;
		}
	} else {
		/* Month Bars */
		timestruct = localtime((time_t *)&sample_begin);
		timestruct->tm_sec = 0;
		timestruct->tm_min = 0;
		timestruct->tm_hour = 0;
		timestruct->tm_mday = 1;
		timestruct->tm_mon--; /* Start the month before the sample */
		marktime = mktime(timestruct);

		x = (marktime-sample_begin)*(((DOUBLE)(XWIDTH-XOFFSET)) / config.range) + XOFFSET;
		while(x < XOFFSET) {
			timestruct->tm_mon++;
			marktime = mktime(timestruct);
			x = (marktime-sample_begin)*((XWIDTH-XOFFSET)/config.range) + XOFFSET;
		}

		while(x < (XWIDTH-10)) {
			/* month lines */
			gdImageLine(im, x, 0, x, YHEIGHT-YOFFSET, grey);
			gdImageLine(im, x+1, 0, x+1, YHEIGHT-YOFFSET, grey);

			mt = (time_t) marktime;
			timestruct = localtime(&mt);
			strftime(buffer, X_LEGEND, "%b", timestruct);
			gdImageString(im, gdFontSmall, x-6,
				YHEIGHT-YOFFSET+10, (PUCHAR) buffer, black);

			/* calculate next x */
			timestruct->tm_mon++;
			marktime = mktime(timestruct);
			x = (marktime-sample_begin)*((XWIDTH-XOFFSET)/config.range) + XOFFSET;
		}
	}

	/* Write the tick marks */

	timestruct = localtime((time_t *)&sample_begin);
	timestruct->tm_sec = 0;
	timestruct->tm_min = 0;
	timestruct->tm_hour = 0;
	marktime = mktime(timestruct);

	if((6*60*60*(XWIDTH-XOFFSET))/config.range > 10)
				/* pixels per 6 hours is more than 2 */
		marktimestep = 6*60*60; /* Major ticks are 6 hours */
	else if((24*60*60*(XWIDTH-XOFFSET))/config.range > 10)
		marktimestep = 24*60*60; /* Major ticks are 24 hours; */
	else
		return;

	x = (marktime-sample_begin)*((XWIDTH-XOFFSET)/config.range) + XOFFSET;
	while(x < XOFFSET) {
		marktime += marktimestep;
		x = (marktime-sample_begin)*((XWIDTH-XOFFSET)/config.range) +
			XOFFSET;
	}

	while(x < (XWIDTH-10)) {
		if(x > XOFFSET) {
			gdImageLine(im, x, YHEIGHT-YOFFSET-5, x,
				YHEIGHT-YOFFSET+5, black);
			gdImageLine(im, x+1, YHEIGHT-YOFFSET-5, x+1,
				YHEIGHT-YOFFSET+5, black);
		}
		marktime += marktimestep;
		x = (marktime-sample_begin)*((XWIDTH-XOFFSET)/config.range) + XOFFSET;
	}

	timestruct = localtime((time_t *)&sample_begin);
	timestruct->tm_sec = 0;
	timestruct->tm_min = 0;
	timestruct->tm_hour = 0;
	marktime = mktime(timestruct);

	if((60*60*(XWIDTH-XOFFSET))/config.range > 2)
					/* pixels per hour is more than 2 */
		marktimestep = 60*60; 	/* Minor ticks are 1 hour */
	else if((6*60*60*(XWIDTH-XOFFSET))/config.range > 2)
		marktimestep = 6*60*60;	/* Minor ticks are 6 hours */
	else if((24*60*60*(XWIDTH-XOFFSET))/config.range > 2)
		marktimestep = 24*60*60;
	else
		return;

	/* draw minor tick marks */

	x = (marktime-sample_begin)*((XWIDTH-XOFFSET)/config.range) + XOFFSET;

	while(x < XOFFSET) {
		marktime += marktimestep;
		x = (marktime-sample_begin)*((XWIDTH-XOFFSET)/config.range) +
			XOFFSET;
	}

	while(x < (XWIDTH-10)) {
		if(x > XOFFSET) {
			gdImageLine(im, x, YHEIGHT-YOFFSET,
				x, YHEIGHT-YOFFSET+5, black);
			gdImageLine(im, x+1, YHEIGHT-YOFFSET, x+1,
				YHEIGHT-YOFFSET+5, black);
		}
		marktime += marktimestep;
		x = (marktime-sample_begin)*((XWIDTH-XOFFSET)/config.range) +
			XOFFSET;
	}
}

/*
 * End of file: graph.c
 *
 */
