/*
 * File: lockctl.c
 *
 * Control program for door lock release.
 *
 * Simply drives the relay with some additional timing and configuration
 * functions.
 *
 * Bob Eager   May 2014
 *
 */

/*
 * History:
 *
 *	1.0	Initial version.
 *
 */

#include "lockctl.h"

/* Forward references */

static	VOID	putusage(VOID);

/* Local storage */

static	CHAR	progname[50];		/* Name of program, as a string */

/* Help text */

static	const	PCHAR helpinfo[] = {
"%s: lock control",
"Synopsis: %s [options] action",
" Options:",
"    -h           display this help",
"    -p port      set port to use (no default)",
"    -t interval  set relock interval (default is %2$d)",
"    -v           be verbose; display status messages",
" ",
"action is 'unlock' or 'lock'.",
" ",
"The relock interval must be between %3$d and %d, expressed in seconds.",
""
};


/*
 * Parse arguments and handle options.
 *
 */

INT main(INT argc, PCHAR argv[])
{	INT ch;
	INT interval = -1;
	BOOL verbose = FALSE;
	BOOL rc;
	ACTION action = ACT_NONE;
	CHAR port[50];

	strcpy(progname, basename(argv[0]));

	/* Process input options */

	port[0] = '\0';
	opterr = 0;			/* Suppress messages */
	while((ch = getopt(argc, argv, ":hp:t:v")) != -1) {
		switch(ch) {
			case 'h':	/* Display help */
				putusage();
				exit(EXIT_SUCCESS);
				break;

			case 'p':	/* Port name */
				if(port[0] != '\0') {
					error(  "port specified"
						" more than once");
					exit(EXIT_FAILURE);
				}
				strncpy(port, optarg, sizeof(port)-1);
				port[sizeof(port)-1] = '\0';
				break;

			case 't':	/* Lock interval */
				if(interval != -1) {
					error(  "relock interval specified"
						" more than once");
					exit(EXIT_FAILURE);
				}
				interval = (INT) strtol(
							optarg,
							(char **) NULL,
							10);
				break;

			case 'v':	/* Verbose - display status */
				verbose = TRUE;
				break;

			case ':':	/* Missing value */
				error("missing value after -%c", optopt);
				exit(EXIT_FAILURE);

			case '?':	/* Invalid flag letter */
				error("invalid flag -%c", optopt);
				exit(EXIT_FAILURE);

			default:
				error("use -h for help");
				exit(EXIT_FAILURE);
		}
	}
	argc -= optind;			/* Move to non-flag arguments */
	argv += optind;

	/* Validate supplied options and apply defaults */

	if(port[0] == '\0') {
		error("port must be specified");
		exit(EXIT_FAILURE);
	}

	if(interval == -1) {
		interval = DEFAULT_RELOCK;
	} else {
		if(interval < MIN_RELOCK || interval > MAX_RELOCK) {
			error("value for -t must be between %d and %d",
				MIN_RELOCK, MAX_RELOCK);
			exit(EXIT_FAILURE);
		}
	}

	/* Check the action argument */

	if(argc == 0) {
		error("no action specified");
		exit(EXIT_FAILURE);
	}
	if(argc > 1) {
		error("only one action permitted");
		exit(EXIT_FAILURE);
	}
	if(strcasecmp(argv[0], "lock") == 0)
		action = ACT_LOCK;
	else if(strcasecmp(argv[0], "unlock") == 0)
		action = ACT_UNLOCK;
	else
		error("unrecognised action specified");

	/* Do the work */

	rc = act(port, interval, action, verbose);

	/* Tidy up and exit */

	return(rc == TRUE ? EXIT_SUCCESS : EXIT_FAILURE);
}


/*
 * Print message on standard error in printf style,
 * accompanied by program name.
 *
 */

VOID error(PCHAR mes, ...)
{	va_list ap;

	fprintf(stderr, "%s: ", progname);

	va_start(ap, mes);
	vfprintf(stderr, mes, ap);
	va_end(ap);

	fputc('\n', stderr);
}


/*
 * Output program usage information.
 *
 */

static VOID putusage(VOID)
{	PCHAR *p = (PCHAR *) helpinfo;
	PCHAR q;

	for(;;) {
		q = *p++;
		if(*q == '\0') break;

		fprintf(stderr, q, progname, DEFAULT_RELOCK, MIN_RELOCK, MAX_RELOCK);
		fputc('\n', stderr);
	}
	fprintf(stderr, "\nThis is version %d.%d\n", VERSION, EDIT);
}

/*
 * End of file: lockctl.c
 *
 */
