/*
 * File: config.c
 *
 * Weather utility
 *
 * Configuration file handler
 *
 * Bob Eager   May 2017
 *
 */

#include "owrep.h"
#include "confcmds.h"

/* Forward references */

static	VOID	config_error(INT, PCHAR, ...);
static	INT	getcmd(PCHAR);


/*
 * Read and parse the configuration file specified by 'configfile'.
 *
 * Returns:
 *	Number of errors encountered.
 *	Any error messages have already been issued.
 *
 * The configuration information is returned in the structure 'config'.
 *
 */

INT read_config(PCHAR configfile, PCONFIG config)
{	INT errors = 0;
	INT line = 0;
	PCHAR p, q, r, s, temp;
	FILE *fp;
	CHAR buf[MAXLINE];

	/* Set defaults */

	memset(config, 0, sizeof(CONFIG));
	config->url = (PCHAR) NULL;
	config->location = 0;
	config->units = UNITS_NONE;
	config->appid = (PCHAR) NULL;

	fp = fopen(configfile, "r");
	if(fp == (FILE *) NULL) {
		config_error(0, "cannot open configuration file %s", configfile);
		return(++errors);
	}

	for(;;) {
		p = fgets(buf, MAXLINE, fp);
		if(p == (PCHAR) NULL) break;
		temp = p + strlen(p) - 1;	/* Point to last character */
		if(*temp == '\n') *temp = '\0';	/* Remove any newline */
		line++;

		p = strchr(buf, '#');		/* Strip comments */
		if(p != (PCHAR) NULL) *p = '\0';

		p = strtok(buf, " \t");
		q = strtok(NULL, " \t");
		r = strtok(NULL, " \t");
		s = strtok(NULL, " \t");

		/* Skip non-information lines */

		if((p == (PCHAR) NULL) ||	/* No tokens */
		   (*p == '\n'))		/* Empty line */
			continue;

		switch(getcmd(p)) {
			case CMD_URL:
				if(config->url != (PCHAR) NULL) {
					config_error(
						line,
						"URL specified more"
						" than once");
					errors++;
					continue;
				}
				if(r != (PCHAR) NULL) {
					config_error(
						line,
						"syntax error (extra on end)");
					errors++;
					continue;
				}
				if(q == (PCHAR) NULL) {
					config_error(
						line,
						"no URL after "
						"URL command");
					errors++;
					break;
				}
				strlwr(q);
				if(strncmp(
						q,
						HTTP_SCHEME,
						strlen(HTTP_SCHEME)) != 0) {
					config_error(
						line,
						"bad scheme in URL"
						" value");
					errors++;
					break;
				}
				p = xmalloc(strlen(q)+1);
				strcpy(p, q);
				config->url = p;
				break;

			case CMD_LOCATION:
				if(config->location != 0) {
					config_error(
						line,
						"LOCATION specified more"
						" than once");
					errors++;
					continue;
				}
				if(r != (PCHAR) NULL) {
					config_error(
						line,
						"syntax error (extra on end)");
					errors++;
					continue;
				}
				if(q == (PCHAR) NULL) {
					config_error(
						line,
						"no value after "
						"LOCATION command");
					errors++;
					break;
				}
				config->location = (INT) strtol(q, NULL, 10);
				break;


			case CMD_UNITS:
				if(config->units != UNITS_NONE) {
					config_error(
						line,
						"UNITS specified more"
						" than once");
					errors++;
					continue;
				}
				if(r != (PCHAR) NULL) {
					config_error(
						line,
						"syntax error (extra on end)");
					errors++;
					continue;
				}
				if(q == (PCHAR) NULL) {
					config_error(
						line,
						"no value after "
						"UNITS command");
					errors++;
					break;
				}
				if(strcasecmp(q, "imperial") == 0) config->units = IMPERIAL;
				if(strcasecmp(q, "metric") == 0) config->units = METRIC;
				if(config->units == UNITS_NONE) {
					config_error(
						line,
						"invalid value "
						"for UNITS command");
					errors++;
				}
				break;

			case CMD_APPID:
				if(config->appid != (PCHAR) NULL) {
					config_error(
						line,
						"APPID specified more"
						" than once");
					errors++;
					continue;
				}
				if(r != (PCHAR) NULL) {
					config_error(
						line,
						"syntax error (extra on end)");
					errors++;
					continue;
				}
				if(q == (PCHAR) NULL) {
					config_error(
						line,
						"no value after "
						"APPID command");
					errors++;
					break;
				}
				p = xmalloc(strlen(q)+1);
				strcpy(p, q);
				config->appid = p;
				break;

			default:
				config_error(
					line,
					"unrecognised command '%s'", p);
				errors++;
				break;
		}
	}

	fclose (fp);

	if(config->url == (PCHAR) NULL) {
		config_error(0, "a URL entry must be given");
		errors++;
	}

	if(config->units == UNITS_NONE) config->units = METRIC;


	if(config->appid == (PCHAR) NULL) {
		config_error(0, "an APPID entry must be given");
		errors++;
	}

	return(errors);
}


/*
 * Check command in 's' for validity, and return command code.
 * Case is immaterial.
 *
 * Returns CMD_BAD if command not recognised.
 *
 */

static INT getcmd(PCHAR s)
{	INT i;

	for(i = 0; ; i++) {
		if(cmdtab[i].cmdcode == CMD_BAD) return(CMD_BAD);
		if(strcasecmp(s, cmdtab[i].cmdname) == 0) break;
	}

	return(cmdtab[i].cmdcode);
}


/*
 * Output configuration error message to standard error in printf style.
 *
 */

static VOID config_error(INT line, PCHAR mes, ...)
{	va_list ap;
	CHAR buf[MAXLOG];

	va_start(ap, mes);
	vsprintf(buf, mes, ap);
	va_end(ap);

	if(line == 0)
		error("config: %s", buf);
	else
		error("config: line %d: %s", line, buf);
}

/*
 * End of file: config.c
 *
 */
