/*
 * File: netio.c
 *
 * General network I/O and timeout routines.
 *
 * Bob Eager   May 2017
 *
 */

#include <sys/select.h>
#include <unistd.h>
#include <stdlib.h>
#include <string.h>
#include <sys/types.h>
#include <sys/socket.h>

#include "netio.h"

#define	BUFSIZE		1024		/* Size of network input buffer */

/* Forward references */

static	INT	fill_buffer(INT, INT);
static	INT	sock_send(INT, PCHAR, INT, INT);

/* Local storage */

static	INT	count;			/* Bytes remaining in input buffer */
static	INT	next;			/* Offset of next byte in input buffer */
static	CHAR	buf[BUFSIZE];		/* Network input buffer */


/*
 * Initialise buffering, etc.
 * Returns:
 *	TRUE		success
 *	FALSE		failure
 *
 */

BOOL netio_init(VOID)
{	/* Initialise count of bytes in network input buffer */

	count = 0;

	return(TRUE);
}


/*
 * Get a line from a socket. Carriage return, linefeed sequence is replaced
 * by a linefeed.
 *
 * Returns:
 *	>= 0			length of line read
 *	SOCKIO_TOOLONG		line too long for buffer; rest of line absorbed
 *	SOCKIO_TIMEOUT		input timed out
 *	SOCKIO_ERR		nonspecific network read error
 *
 */

INT sock_gets(PCHAR line, INT size, INT sockno, INT timeout)
{	INT len = 0;
	CHAR c;
	BOOL full = FALSE;

	for(;;) {
		if(count == 0) count = fill_buffer(sockno, timeout);
		if(count == 0) return(SOCKIO_ERR);
		if(count < 0) return(SOCKIO_TIMEOUT);

		c = buf[next++];
		count--;
		if(c == '\r') {
			if(count == 0) count = fill_buffer(sockno, timeout);
			if(count == 0) return(SOCKIO_ERR);
			if(count < 0) return(SOCKIO_TIMEOUT);

			if(buf[next] == '\n') {
				next++;
				count--;
				if(full == FALSE) line[len++] = '\n';
				break;
			}
		}
		if(full == FALSE) line[len++] = c;
		if(len == size - 1) full = TRUE;
		if(c == '\n') break;
	}
	
	line[len] = '\0';

	return(full ? SOCKIO_TOOLONG : len);
}


/*
 * Get a line from a socket. Carriage returns and linefeeds are
 * left alone.
 *
 * Returns:
 *	>= 0			length of line read
 *	SOCKIO_TOOLONG		line too long for buffer; rest of line absorbed
 *	SOCKIO_TIMEOUT		input timed out
 *	SOCKIO_ERR		nonspecific network read error
 *
 */

INT sock_gets_raw(PCHAR line, INT size, INT sockno, INT timeout)
{	INT len = 0;
	CHAR c;
	BOOL full = FALSE;

	for(;;) {
		if(count == 0) count = fill_buffer(sockno, timeout);
		if(count == 0) return(SOCKIO_ERR);
		if(count < 0) return(SOCKIO_TIMEOUT);

		c = buf[next++];
		count--;
		if(full == FALSE) line[len++] = c;
		if(len == size - 1) full = TRUE;
		if((c == '\n') || (count == 0)) break;
	}
	
	line[len] = '\0';

	return(full ? SOCKIO_TOOLONG : len);
}


/*
 * Send a line to a socket. Massages a terminating linefeed (\n)
 * into carriage return followed by linefeed.
 *
 */

VOID sock_puts(PCHAR line, INT sockno, INT timeout)
{	static const CHAR crlf[] = "\r\n";
	INT len = strlen(line);

	if(line[len-1] == '\n') {
		len--;
		sock_send(sockno, line, len, timeout);
		len = strlen(crlf);
		line = (PCHAR) &crlf[0];
	}
	sock_send(sockno, line, len, timeout);
}


/*
 * Refill the network input buffer.
 *
 * Returns:
 *	>0		number of bytes in buffer
 *	0		nonspecific network read error
 *	<0		timeout
 *
 */

static INT fill_buffer(INT sockno, INT timeout)
{	INT rc;
	INT len;
	INT nfds = sockno + 1;
	fd_set rfds, xfds;;
	struct timeval t;

	next = 0;			/* Reset buffer pointer */

	/* Set up and perform select call */

	FD_ZERO(&rfds); FD_ZERO(&xfds);
	FD_SET(sockno, &rfds);		/* Sockets for read check */
	FD_SET(sockno, &xfds);		/* Sockets for exception check */

	t.tv_sec = (long) timeout;
	t.tv_usec = 0;

	rc = select(nfds, &rfds, (fd_set *) 0, &xfds, &t) ;

	if(rc == 0) return(-1);		/* Timeout expired */
	if(rc < 0) return(0);		/* Error */

	if(FD_ISSET(sockno, &xfds))	/* Exception on socket */
		return(0);

	if(FD_ISSET(sockno, &rfds)) {	/* Read ready */
		len = recv(sockno, buf, BUFSIZE, 0);
		return(len);
	}

	return(0);			/* Some other problem */
}


/*
 * Write a buffer to a socket.
 *
 * Returns:
 *	same as for 'send'
 *
 */

static INT sock_send(INT sockno, PCHAR buf, INT len, INT timeout)
{	return(send(sockno, buf, len, 0));
}

/*
 * End of file: netio.c
 *
 */
