/*
 * File: extract.c
 *
 * Weather utility
 *
 * XML data extraction
 *
 * Bob Eager   May 2017
 *
 */

#include "owrep.h"
#include <expat.h>

/* Constants */

#define		STACKSIZE	10	/* Size of internal state stack */
#define		MAXATTRTEXT	200	/* Length of an attribute's text part */

/* Type definitions */

typedef enum {				/* State values */
ST_NEUTRAL, ST_CURRENT, ST_CITY, ST_COORD, ST_COUNTRY, ST_SUN, ST_TEMP,
ST_HUMIDITY, ST_PRESSURE, ST_WIND, ST_SPEED, ST_GUSTS, ST_DIRECTION,
ST_CLOUDS, ST_VISIBILITY, ST_PRECIPITATION, ST_LASTUPDATE
} STATE;

typedef struct _ATTR {			/* Attribute storage */
CHAR	text[MAXATTRTEXT+1];		/* Text component */
UINT	len;				/* Text length */
BOOL	quiet;				/* TRUE for no more messages */
} ATTR, *PATTR;

/* Forward references */

static	VOID	char_handler(PVOID, const XML_Char *, INT);
static	VOID	end_handler(PVOID, const XML_Char *);
static	VOID	start_handler(PVOID, const XML_Char *, const XML_Char **);
static	VOID	pop_state(STATE);
static	VOID	push_state(STATE, STATE);

/* Local storage */

static	STATE	state;			/* Current parse state */
static	STATE	stack[STACKSIZE];	/* Tag stack */
static	INT	stkptr;			/* Tag stack pointer */
static	ATTR	a_country;		/* Country attributes */


/*
 * Extract weather details, from the raw XML.
 *
 *	Inputs:
 *		buf	buffer containing XML
 *		len	length of data in buffer
 *		config	pointer to configuration structure
 *		weather	pointer to weather structure for output
 *
 *	Outputs:
 *		weather structure filled in; if 'valid' field is FALSE,
 *		the extraction failed.
 *
 */

VOID extract_weather(PCHAR buf, INT len, PCONFIG config, PWEATHER weather)
{	INT rc;
	XML_Parser xp;

	weather->valid = FALSE;		/* Pro tem */
	weather->config = config;
	strcpy(weather->cityname, "UNKNOWN");
	strcpy(weather->country, "UNKNOWN");
	weather->coord_lon = 0.0;
	weather->coord_lat = 0.0;
	weather->temp_now = 0;
	weather->temp_min = 0;
	weather->temp_max = 0;
	weather->humidity = 0;
	weather->humidity_units[0] = '\0';
	weather->pressure = 0;
	weather->pressure_units[0] = '\0';
	weather->wind_speed = 0.0;
	weather->wind_speed_units[0] = '\0';	/* Filled in manually */
	weather->wind_speed_gusts = 0.0;
	weather->wind_speed_name[0] = '\0';
	weather->wind_direction = 0;
	weather->wind_direction_code[0] = '\0';
	weather->wind_direction_name[0] = '\0';
	weather->clouds = 0;
	weather->clouds_name[0] = '\0';
	weather->visibility = 0;
	weather->visibility_units[0] = '\0';	/* Filled in manually */
	weather->precipitation = -1.0;
	weather->precipitation_mode[0] = '\0';
	weather->precipitation_units[0] = '\0';	/* Filled in manually */

	/* Set up for parsing */

	xp = XML_ParserCreate(NULL);
	if(xp == NULL) return;
	XML_SetUserData(xp, weather);
	XML_SetElementHandler(xp, &start_handler, &end_handler);
	XML_SetCharacterDataHandler(xp, &char_handler);
	state = ST_NEUTRAL;
	stkptr = 0;

	/* Do the parse */

	rc = XML_Parse(xp, buf, len, TRUE);

	/* Clean up the parser */

	XML_ParserFree(xp);

	if(rc == 0) return;		/* parse failed */

	weather->valid = TRUE;
}


/*
 * Start tag element handler.
 * This is a callback from the XML parser, and is invoked when a start tag
 * is processed.
 *
 *	Inputs:
 *		vw	VOID pointer to weather structure
 *		name	XML tag name
 *		attr	pointer to array of attributes
 *
 *	Outputs:
 *		none
 *
 */

static VOID start_handler(PVOID vw, const XML_Char *name,
		const XML_Char ** attr)
{	PWEATHER weather = (PWEATHER) vw;
	const XML_Char **p, **q;

	if(weather->config->debug == TRUE)
		fprintf(stderr, "\nHandling <%s>\n", name);

	if(strcasecmp(name, "Current") == 0) {
		push_state(ST_CURRENT, ST_NEUTRAL);
		return;
	}
	if(strcasecmp(name, "City") == 0) {
		push_state(ST_CITY, ST_CURRENT);
		p = attr;
		while(*p != (XML_Char *) NULL) {
			q = p;
			p++;
			if(strcasecmp(*q, "Name") == 0)
				strcpy(weather->cityname, *p);
			p++;
		}
		return;
	}
	if(strcasecmp(name, "Coord") == 0) {
		push_state(ST_COORD, ST_CITY);
		p = attr;
		while(*p != (XML_Char *) NULL) {
			q = p;
			p++;
			if(strcasecmp(*q, "Lon") == 0)
				weather->coord_lon = strtof(*p, NULL);
			if(strcasecmp(*q, "Lat") == 0)
				weather->coord_lat = strtof(*p, NULL);
			p++;
		}
		return;
	}
	if(strcasecmp(name, "Country") == 0) {
		a_country.len = 0;
		a_country.quiet = FALSE;
		push_state(ST_COUNTRY, ST_CITY);
		return;
	}
	if(strcasecmp(name, "Sun") == 0) {
		push_state(ST_SUN, ST_CITY);
		p = attr;
		while(*p != (XML_Char *) NULL) {
			q = p;
			p++;
			if(strcasecmp(*q, "Rise") == 0)
				strcpy(weather->sun_rise, *p);
			if(strcasecmp(*q, "Set") == 0)
				strcpy(weather->sun_set, *p);
			p++;
		}
		return;
	}
	if(strcasecmp(name, "Temperature") == 0) {
		push_state(ST_TEMP, ST_CURRENT);
		p = attr;
		while(*p != (XML_Char *) NULL) {
			q = p;
			p++;
			if(strcasecmp(*q, "Value") == 0)
				weather->temp_now = (INT) strtol(*p, NULL, 10);
			if(strcasecmp(*q, "Min") == 0)
				weather->temp_min = (INT) strtol(*p, NULL, 10);
			if(strcasecmp(*q, "Max") == 0)
				weather->temp_max = (INT) strtol(*p, NULL, 10);
			if(strcasecmp(*q, "Unit") == 0)
				strcpy(weather->temp_units, *p);
			p++;
		}
		return;
	}
	if(strcasecmp(name, "Humidity") == 0) {
		push_state(ST_HUMIDITY, ST_CURRENT);
		p = attr;
		while(*p != (XML_Char *) NULL) {
			q = p;
			p++;
			if(strcasecmp(*q, "Value") == 0)
				weather->humidity = (INT) strtol(*p, NULL, 10);
			if(strcasecmp(*q, "Unit") == 0)
				strcpy(weather->humidity_units, *p);
			p++;
		}
		return;
	}
	if(strcasecmp(name, "Pressure") == 0) {
		push_state(ST_PRESSURE, ST_CURRENT);
		p = attr;
		while(*p != (XML_Char *) NULL) {
			q = p;
			p++;
			if(strcasecmp(*q, "Value") == 0)
				weather->pressure = (INT) strtol(*p, NULL, 10);
			if(strcasecmp(*q, "Unit") == 0)
				strcpy(weather->pressure_units, *p);
			p++;
		}
		return;
	}
	if(strcasecmp(name, "Wind") == 0) {
		push_state(ST_WIND, ST_CURRENT);
		return;
	}
	if(strcasecmp(name, "Speed") == 0) {
		push_state(ST_SPEED, ST_WIND);
		p = attr;
		while(*p != (XML_Char *) NULL) {
			q = p;
			p++;
			if(strcasecmp(*q, "Value") == 0)
				weather->wind_speed = strtof(*p, NULL);
			if(strcasecmp(*q, "Name") == 0)
				strcpy(weather->wind_speed_name, *p);
			p++;
		}
		return;
	}
	if(strcasecmp(name, "Gusts") == 0) {
		push_state(ST_GUSTS, ST_WIND);
		p = attr;
		while(*p != (XML_Char *) NULL) {
			q = p;
			p++;
			if(strcasecmp(*q, "Value") == 0)
				weather->wind_speed_gusts = strtof(*p, NULL);
			p++;
		}
		return;
	}
	if(strcasecmp(name, "Direction") == 0) {
		push_state(ST_DIRECTION, ST_WIND);
		p = attr;
		while(*p != (XML_Char *) NULL) {
			q = p;
			p++;
			if(strcasecmp(*q, "Value") == 0)
				weather->wind_direction = (INT) strtol(*p, NULL, 10);
			if(strcasecmp(*q, "Code") == 0)
				strcpy(weather->wind_direction_code, *p);
			if(strcasecmp(*q, "Name") == 0)
				strcpy(weather->wind_direction_name, *p);
			p++;
		}
		return;
	}
	if(strcasecmp(name, "Clouds") == 0) {
		push_state(ST_CLOUDS, ST_CURRENT);
		p = attr;
		while(*p != (XML_Char *) NULL) {
			q = p;
			p++;
			if(strcasecmp(*q, "Value") == 0)
				weather->clouds = (INT) strtol(*p, NULL, 10);
			if(strcasecmp(*q, "Name") == 0)
				strcpy(weather->clouds_name, *p);
			p++;
		}
		return;
	}
	if(strcasecmp(name, "Visibility") == 0) {
		push_state(ST_VISIBILITY, ST_CURRENT);
		p = attr;
		while(*p != (XML_Char *) NULL) {
			q = p;
			p++;
			if(strcasecmp(*q, "Value") == 0)
				weather->visibility = (INT) strtol(*p, NULL, 10);
			p++;
		}
		return;
	}
	if(strcasecmp(name, "Precipitation") == 0) {
		push_state(ST_PRECIPITATION, ST_CURRENT);
		p = attr;
		while(*p != (XML_Char *) NULL) {
			q = p;
			p++;
			if(strcasecmp(*q, "Value") == 0)
				weather->precipitation = strtof(*p, NULL);
			if(strcasecmp(*q, "Mode") == 0)
				strcpy(weather->precipitation_mode, *p);
			p++;
		}
		return;
	}
	if(strcasecmp(name, "Lastupdate") == 0) {
		push_state(ST_LASTUPDATE, ST_CURRENT);
		p = attr;
		while(*p != (XML_Char *) NULL) {
			q = p;
			p++;
			if(strcasecmp(*q, "Value") == 0)
				strcpy(weather->lastupdate, *p);
			p++;
		}
		return;
	}
}


/*
 * End tag element handler.
 * This is a callback from the XML parser, and is invoked when an end tag
 * is processed.
 *
 *	Inputs:
 *		vw	VOID pointer to weather structure
 *		name	XML tag name
 *
 *	Outputs:
 *		none
 *
 */

static VOID end_handler(PVOID vw, const XML_Char *name)
{	PWEATHER weather = (PWEATHER) vw;

	if(weather->config->debug == TRUE)
		fprintf(stderr, "\nHandling </%s>\n", name);

	if(strcasecmp(name, "Current") == 0) {
		pop_state(ST_NEUTRAL);
		return;
	}
	if(strcasecmp(name, "City") == 0) {
		pop_state(ST_CURRENT);
		return;
	}
	if(strcasecmp(name, "Coord") == 0) {
		pop_state(ST_CITY);
		return;
	}
	if(strcasecmp(name, "Country") == 0) {
		a_country.text[a_country.len] = '\0';
		strcpy(weather->country, a_country.text);
		pop_state(ST_CITY);
		return;
	}
	if(strcasecmp(name, "Sun") == 0) {
		pop_state(ST_CITY);
		return;
	}
	if(strcasecmp(name, "Temperature") == 0) {
		pop_state(ST_CURRENT);
		return;
	}
	if(strcasecmp(name, "Humidity") == 0) {
		pop_state(ST_CURRENT);
		return;
	}
	if(strcasecmp(name, "Pressure") == 0) {
		pop_state(ST_CURRENT);
		return;
	}
	if(strcasecmp(name, "Wind") == 0) {
		pop_state(ST_CURRENT);
		return;
	}
	if(strcasecmp(name, "Speed") == 0) {
		pop_state(ST_WIND);
		return;
	}
	if(strcasecmp(name, "Gusts") == 0) {
		pop_state(ST_WIND);
		return;
	}
	if(strcasecmp(name, "Direction") == 0) {
		pop_state(ST_WIND);
		return;
	}
	if(strcasecmp(name, "Clouds") == 0) {
		pop_state(ST_CURRENT);
		return;
	}
	if(strcasecmp(name, "Visibility") == 0) {
		pop_state(ST_CURRENT);
		return;
	}
	if(strcasecmp(name, "Precipitation") == 0) {
		pop_state(ST_CURRENT);
		return;
	}
	if(strcasecmp(name, "Lastupdate") == 0) {
		pop_state(ST_CURRENT);
		return;
	}
}


/*
 * Character data handler.
 *
 */

static VOID char_handler(PVOID vw, const XML_Char *s, INT len)
{	PWEATHER weather = (PWEATHER) vw;
	INT i;

	if(weather->config->debug == TRUE && len > 1) {
		for(i = 0; i < len; i++)
			fputc(s[i], stderr);
	}

	switch(state) {
		case ST_NEUTRAL:
		case ST_CITY:
		case ST_COORD:
		case ST_SUN:
		case ST_TEMP:
		case ST_HUMIDITY:
		case ST_PRESSURE:
		case ST_WIND:
		case ST_SPEED:
		case ST_GUSTS:
		case ST_DIRECTION:
		case ST_CLOUDS:
		case ST_VISIBILITY:
		case ST_PRECIPITATION:
		case ST_LASTUPDATE:
			break;

		case ST_COUNTRY:
			if(a_country.len + len > MAXATTRTEXT) {
				len = MAXATTRTEXT - a_country.len;
				if(a_country.quiet == FALSE) {
					dolog("country text truncated");
					a_country.quiet = TRUE;
				}
			}
			strncpy(&a_country.text[a_country.len], s, len);
			a_country.len = a_country.len + len;
			break;

		default:		/* Ignore */
			break;
	}
}


/*
 * Stack the old state (fater checking it) and set a new one.
 *
 */

static VOID push_state(STATE newstate, STATE curstate)
{	if(stkptr >= STACKSIZE -1 ) {
		fprintf(stderr, "Fatal error: stack overflow\n");
		exit(EXIT_FAILURE);
	}
	if(state != curstate) {
		error("fatal - moving from state %d, not %d",
			state, curstate);
			exit(EXIT_FAILURE);
	}
	stack[stkptr++] = state;
	state = newstate;
}


/*
 * Unstack the previous state and check it is as expected.
 *
 */

static VOID pop_state(STATE expected_state)
{	if(stkptr == 0) {
		fprintf(stderr, "Fatal error: stack underflow!\n");
		exit(EXIT_FAILURE);
	}
	state = stack[--stkptr];
	if(state != expected_state) {
		error("fatal - unstacked state %d, expected %d",
			state, expected_state);
		exit(EXIT_FAILURE);
	}
}

/*
 * End of file: extract.c
 *
 */
