/*
 * File: lookup.c
 *
 * Weather utility
 *
 * Network lookup
 *
 * Bob Eager   May 2017
 *
 */

#include "owrep.h"
#include "netio.h"

#define	MAXHOST		100		/* Max length of hostname */
#define	MAXMES		2000		/* Max size of HTTP message */
#define	MAXURL		200		/* Max length of query URL */
#define	FLAG		'$'		/* Flag character in URL */
#define	HDR_DEL		" \t\n"		/* Header delimiters */
#define	HDR_DEL2	"\n"		/* Header delimiters to end */

#define	SEND_TIMEOUT	30
#define	RECV_TIMEOUT	30

/* Forward references */

static	VOID	fix_domain(PCHAR);
static	VOID	get_url(PCHAR, PCONFIG);
static	PCHAR	transact(INT, PCHAR, PCHAR, PINT);


/*
 * Get the weather.
 *
 *	Inputs:
 *		config	pointer to configuration structure
 *		datalen	pointer to receive returned data length
 *
 *	Outputs:
 *	If successful, returns pointer to allocated data buffer, length in 'datalen'.
 *	If failed, returns NULL.
 *
 */

PCHAR get_weather(PCONFIG config, PINT datalen)
{	PCHAR p, q, r;
	INT sockno, rc;
	INT l = 0;			/* Initialised to keep compiler quiet */
	CHAR url[MAXURL];
	CHAR temp[MAXURL];
	CHAR lhost[MAXHOST];		/* Lookup hostname */
	PCHAR urlp;			/* Non-host part of URL */
	PHOST host;			/* Host entry for lookup host */
	PSERV serv;			/* Service entry for HTTP */
	SOCK server;
	ULONG lh_addr;			/* IP address of lookup host */
	USHORT port = 0;		/* Port on lookup host */

	r = temp;
	r[0] = '\0';
	get_url(url, config);/* Extract and expand URL */
	p = &url[strlen(HTTP_SCHEME)];	/* Point to host name */
	for(q = p; *q != '\0'; q++) {	/* Find end of host name */
		l = q - p;
		if(*q == ':') {		/* Extract port */
			q++;
			while(*q != '/' && *q != '\0') *r++ = *q++;
			*r = '\0';
			break;
		}
		if(*q == '/') break;
	}
	urlp = q;			/* Point after the host/port */

	strncpy(lhost, p, l);
	lhost[l] = '\0';
	fix_domain(lhost);

	/* 'lhost' contains the server name, and 'temp' is either null,
	   or contains a port number.
	*/

	/* Look up IP address for the lookup host */

	host = gethostbyname(lhost);
	if(host == (PHOST) NULL) {
		if(isdigit(lhost[0])) {
			lh_addr = inet_addr(lhost);
		} else {
			error(
				"cannot get address for lookup host '%s'",
				lhost);
			return(FALSE);
		}
	} else {
		lh_addr = *((PULONG) host->h_addr);
	}

	/* Now set the port, either as requested or by default */

	if(temp[0] != '\0') {		/* Port specified */
		port = htons(atoi(temp));
	} else {
		serv = getservbyname(HTTPSERVICE, TCP);
		if(serv == (PSERV) NULL) {
			error("cannot get port for %s/%s service", HTTPSERVICE, TCP);
			return(FALSE);
		}
		port = serv->s_port;
		endservent();
	}

	/* Get a socket, set it up and try to connect */

	sockno = socket(PF_INET, SOCK_STREAM, 0);
	if(sockno == -1) {
		error("cannot create socket");
		return(FALSE);
	}
	memset(&server, 0, sizeof(server));
	server.sin_family = AF_INET;
	server.sin_addr.s_addr = lh_addr;
	server.sin_port = port;
	rc = connect(sockno, (PSOCKG) &server, sizeof(SOCK));
	if(rc == -1) {
		error("cannot connect to server '%s'", lhost);
		return(FALSE);
	}

	/* Carry out the transaction with the server */

	p = transact(sockno, urlp, lhost, &l);
	close(sockno);

	*datalen = l;
	return(p);
}


/*
 * Carry out the transaction with the server.
 *
 *	Inputs:
 *		sockno	socket number to use
 *		url	full URL
 *		lhost	lookup hostname
 *		datalen	pointer to place to return data length
 *
 *	Outputs:
 *	If successful, returns pointer to data buffer, length in 'datalen'.
 *	If failed, returns NULL.
 *
 */

static PCHAR transact(INT sockno, PCHAR url, PCHAR lhost, PINT datalen)
{	INT len, size, temp;
	INT count;
	LONG clen;
	CHAR buf[MAXMES];
	PCHAR p, q, r;
	PCHAR data = (PCHAR) NULL;
	BOOL chunked = FALSE;

	/* Reset the network I/O module */

	netio_init();

	/* Construct and send the HTTP GET message */

	/* First, the method part */

	sprintf(buf, "GET %s HTTP/1.1\n", url);
	sock_puts(buf, sockno, SEND_TIMEOUT);

	/* Now the headers */

	sprintf(buf, "Host: %s\n", lhost);
	sock_puts(buf, sockno, SEND_TIMEOUT);

	sprintf(buf, "User-Agent: weather/%d.%d\n", VERSION, EDIT);
	sock_puts(buf, sockno, SEND_TIMEOUT);

	strcpy(buf, "Accept: text/xml,application/xml,application/xhtml+xml"
		    ",text/html,text/plain\n");
	sock_puts(buf, sockno, SEND_TIMEOUT);

	strcpy(buf, "Accept-Encoding:\n");
	sock_puts(buf, sockno, SEND_TIMEOUT);

	strcpy(buf, "Connection: close\n");
	sock_puts(buf, sockno, SEND_TIMEOUT);

	/* Blank line for end of headers */

	strcpy(buf, "\n");
	sock_puts(buf, sockno, SEND_TIMEOUT);

	/* Now read the response */

	len = sock_gets(buf, MAXMES, sockno, RECV_TIMEOUT);
	if(len < 0) {
		error("network read error");
		return((PCHAR) NULL);
	}

	p = strtok(buf, HDR_DEL);		/* HTTP version */
	q = strtok(NULL, HDR_DEL);		/* Response code */
	r = strtok(NULL, HDR_DEL2);		/* Reason phrase */
	if(q[0] != '2') {
		error("server response was %s (%s)", q, r);
		return((PCHAR) NULL);
	}

	/* Get the rest of the headers */

	clen = 0;				/* Default content length */
	do {
		len = sock_gets(buf, MAXMES, sockno, RECV_TIMEOUT);
		p = strtok(buf, HDR_DEL);
		q = strtok(NULL, HDR_DEL);
		if(p == (PCHAR) NULL || q == (PCHAR) NULL) continue;
		if(strcasecmp(p, "Transfer-Encoding:") == 0) {
			if(strcasecmp(q, "chunked") == 0) {
			chunked = TRUE;
			}
		}
		if(strcasecmp(p, "Content-Length:") == 0) {
			clen = strtol(q, NULL, 10);
		}
	} while(buf[0] != '\n');


	/* Now get the body; the content may be in HTTP 'chunked' format.
	   Each chunk is preceded by its length in hexadecimal.
           The length is separated from the chunk by newline, and
	   from the next chunk count by newline. Counts assume CR-LF
	   between lines.
	   On the other hand, it may just be plain text; its length is then given by
	   the Content-Length header. */

	/* Data are accumulated in a growing buffer, allocated via
	   'realloc'. The caller must free this buffer when it is no
	   longer needed. The chunk must be read 'raw' to avoid any removal
	   of carriage return characters, since they contribute to the
	   quoted size. */

	count = 0;				/* Bytes so far */
	for(;;) {
		if(chunked == TRUE) {
			len = sock_gets(buf, MAXMES, sockno, RECV_TIMEOUT);
						/* Read line with chunk size */
			if(len == 0) return(FALSE); /* Should not happen */
			sscanf(buf, "%x", &size); /* Decode chunk size */
			if(size == 0) break;	/* No more chunks */
		} else size = clen;		/* Not chunked - just one iteration */
		temp = count;			/* Save current data size */
		count += size;			/* Adjust to new total size */
		data = realloc(data, count);	/* Grow the buffer */
		p = data + temp;		/* Beyond end of current data */
		while(size > 0) {		/* Read lines from chunk */
			len = sock_gets_raw(buf, MAXMES, sockno, RECV_TIMEOUT);
			if(len <= 0) break;	/* Should not happen */
			if(len > size) len = size;
			memcpy(p, buf, len);	/* Move to data buffer */
			p += len;		/* Move to next free byte */
			size = size - len;	/* This much left in chunk */
		}
		if(clen != 0) break;
	}
	*datalen = count;

	return(data);
}


/*
 * Build the full URL for the query, using the ISBN and the configuration
 * data.
 *
 *	Inputs:
 *		url	buffer for output URL
 *		config	configuration structure for URL and access key
 *
 *	Outputs:
 *		via inputs, otherwise none
 *
 */

static VOID get_url(PCHAR url, PCONFIG config)
{	PCHAR p, q, r;
	INT c, cu;
	CHAR s[100];

	p = &config->url[0];
	q = &url[0];
	for(;;) {
		c = *p++;
		if(c == '\0') {
			*q = '\0';
			break;
		}
		if(c != FLAG) {
			*q++ = c;
			continue;
		}
		c = *p++;
		if(c == '\0') {
			*q++ = FLAG;
			*q = '\0';
			break;
		}
		cu = toupper(c);
		if(cu == 'A') {		/* Application ID */
			r = &config->appid[0];
			while(*r != '\0') *q++ = *r++;
			continue;
		}
		if(cu == 'U') {		/* Units */
			switch(config->units) {
				case IMPERIAL:
					r = "imperial";
					break;

				case METRIC:
				default:
					r = "metric";
					break;
			}
			while(*r != '\0') *q++ = *r++;
			continue;
		}
		if(cu == 'L') {		/* Location */
			sprintf(s, "%d", config->location);
			r = &s[0];
			while(*r != '\0') *q++ = *r++;
			continue;
		}
		*q++ = c;
	}
}


/*
 * Check for a full domain name; if not present, add default domain name.
 *
 */

static VOID fix_domain(PCHAR name)
{	if(strchr(name, '.') == (PCHAR) NULL && _res.defdname[0] != '\0') {
		strcat(name, ".");
		strcat(name, _res.defdname);
	}
}

/*
 * End of file: lookup.c
 *
 */
