/*
 * File: output.c
 *
 * Weather utility
 *
 * Data output
 *
 * Bob Eager   May 2017
 *
 */

#include "owrep.h"

/* Forward references */

static	VOID	output_compressed(PCONFIG, PWEATHER);
static	VOID	output_human(PWEATHER);


/*
 * Output the data in 'weather', in the requested format.
 *
 *	Inputs:
 *		config	pointer to configuration structure
 *			'compressed' indicates type of output
 *
 *		weather	pointer to weather structure for input and output
 *
 *	Outputs:
 *		none
 *
 */

VOID output_weather(PCONFIG config, PWEATHER weather)
{	switch(config->compressed) {
		case TRUE:
			output_compressed(config, weather);
			break;

		case FALSE:
			output_human(weather);
	}
}


/*
 * Output the data in 'weather', in compressed format.
 *
 *	Inputs:
 *		weather	pointer to weather structure for input and output
 *
 *	Outputs:
 *		weather information to standard output.
 *		Output is one one line, with fields separated by the
 *		specified delimiter; strings (s) or integers (i) or
 *		reals (r) or dates (d) or times (t).
 *
 *		1	s	City name
 *		2	s	Country
 *		3	r	Longitude		
 *		4	s	Longitude direction (E or W)
 *		5	r	Latitude
 *		6	s	Latitude direction (N or S)
 *		7	d	Sunrise date
 *		8	t	Sunrise time
 *		9	d	Sunset date
 *		10	t	Sunset time
 *		11	s	Temperature units
 *		12	i	Temperature now
 *		13	i	Temperature minimum
 *		14	i	Temperature maximum
 *		15	i	Humidity
 *		16	s	Humidity units
 *		17	i	Pressure
 *		18	s	Pressure units
 *		19	s	Wind speed units
 *		20	r	Wind speed
 *		21	s	Wind speed description
 *		22	r	Gust speed
 *		23	i	Wind direction (bearing)
 *		24	s	Wind direction code (e.g. SW)
 *		25	s	Wind direction description
 *		26	i	Cloud cover percentage
 *		27	s	Cloud cover description
 *		28	i	Visibility distance
 *		29	s	Visibility units
 *		30	s	Precipitation mode
 *		31	i	Precipitation
 *		32	s	Precipitation units
 *		33	d	Date of last update
 *		34	t	Time of last update
 *
 */

static VOID output_compressed(PCONFIG config, PWEATHER weather)
{	INT del = config->delim;
	CHAR sun_rise[20], sun_set[20];
	CHAR lastupdate[20];

	strcpy(sun_rise, weather->sun_rise);
	sun_rise[10] = del;
	strcpy(sun_set, weather->sun_set);
	sun_set[10] = del;
	strcpy(lastupdate, weather->lastupdate);
	lastupdate[10] = del;

	printf(
		"%s%c%s%c"
		"%.2f%c%s%c"
		"%.2f%c%s%c"
		"%s%c%s%c"
		"%s%c%d%c"
		"%d%c%d%c"
		"%d%c%s%c"
		"%d%c%s%c"
		"%s%c%.1f%c"
		"%s%c%.1f%c"
		"%d%c%s%c"
		"%s%c"
		"%d%%%c%s%c"
		"%d%c%s%c"
		"%s%c"
		"%.0f%c%s%c"
		"%s%c"
		"\n",
		weather->cityname, del, weather->country, del,
		weather->coord_lon, del, weather->coord_lon_direction, del,
		weather->coord_lat, del, weather->coord_lat_direction, del,
		sun_rise, del, sun_set, del,
		&weather->temp_units[0], del, weather->temp_now, del,
		weather->temp_min, del, weather->temp_max, del,
		weather->humidity, del, &weather->humidity_units[0], del, 
		weather->pressure, del, &weather->pressure_units[0], del,
		weather->wind_speed_units, del, weather->wind_speed, del,
		weather->wind_speed_name, del, weather->wind_speed_gusts, del,
		weather->wind_direction, del, weather->wind_direction_code, del,
		weather->wind_direction_name, del,
		weather->clouds, del, weather->clouds_name, del,
		weather->visibility, del, weather->visibility_units, del,
		weather->precipitation_mode, del,
		weather->precipitation, del, weather->precipitation_units, del,
		lastupdate, del);
}


/*
 * Output the data in 'weather', in human readable format.
 *
 *	Inputs:
 *		weather	pointer to weather structure for input and output
 *
 *	Outputs:
 *		weather information to standard output.
 *
 */

static VOID output_human(PWEATHER weather)
{	printf("City:\t\t%s (%s)\n",
		&weather->cityname[0], &weather->country[0]);
	printf("Longitude:\t%.2f%s\nLatitude:\t%.2f%s\n",
		weather->coord_lon, weather->coord_lon_direction,
		weather->coord_lat, weather->coord_lat_direction);
	printf("Sunrise:\t%s\nSunset\t\t%s\n",
		weather->sun_rise, weather->sun_set);
	printf("Temperature:\t%d %s (min %d, max %d)\n",
		weather->temp_now, &weather->temp_units[0],
		weather->temp_min, weather->temp_max);
	printf("Humidity:\t%d%s\n", weather->humidity,
		&weather->humidity_units[0]); 
	printf("Pressure:\t%d %s\n", weather->pressure,
		&weather->pressure_units[0]);
	printf("Wind:\t\t%.1f %s (%s)\n", weather->wind_speed,
		weather->wind_speed_units,
		weather->wind_speed_name);
	if(weather->wind_speed_gusts != 0.0)
		printf("Gusts:\t\t%.1f %s\n",
		weather->wind_speed_gusts,
		weather->wind_speed_units);
	printf("Wind direction:\t%d",
		weather->wind_direction);
	if(weather->wind_direction_code[0] != '\0')
			printf(" (%s)", 
				weather->wind_direction_code);
	if(weather->wind_direction_name[0] != '\0')
			printf(" (%s)", 
				weather->wind_direction_name);
	fputc('\n', stdout);
	printf("Clouds:\t\t%d%% (%s)\n", weather->clouds,
		weather->clouds_name);
	printf("Visibility:\t%d %s\n",
		weather->visibility, weather->visibility_units);
	printf("Precipitation:\t%s", weather->precipitation_mode);
	if(strcasecmp(&weather->precipitation_mode[0], "no") != 0)
		printf(" %.0f %s",
		weather->precipitation,
		weather->precipitation_units);
	fputc('\n', stdout);
	printf("Last update:\t%s\n", weather->lastupdate);
}

/*
 * End of file: output.c
 *
 */
