/*
 * File owrep.c
 *
 * Weather utility
 *
 * Main program
 *
 * Bob Eager   May 2017
 *
 */

/*
 * History:
 *
 *	1.0	Initial version.
 *	1.1	Fix unrecognised -d flag
 *
 */

#include "owrep.h"

#define	DEFAULT_CONFIG	"/usr/local/etc/owrep.conf"

/* Forward references */

static	VOID		putusage(VOID);
static	BOOL		starts_with(PCHAR, PCHAR);

/* Local storage */

static	CONFIG		config;
static	PCHAR		progname;

/* Help text */

static	const	PCHAR helpinfo[] = {
"%s: Weather utility",
"Synopsis: %s [options]",
" Options:",
"    -c file      specify alternate configuration file",
"    -d delim     sepecify output delimiter (default @)",
"    -D           enable debug output",
"    -h           display this help",
"    -l location  set location (overrides config)",
"    -u units     select units: (i)mperial or (m)etric",
"    -z           generate output in compressed format",
" ",
"Default configuration file is "DEFAULT_CONFIG,
""
};


/*
 * Parse arguments and handle options.
 *
 */

INT main(int argc, char *argv[])
{	INT ch, rc, len;
	INT location = -1;
	CHAR delim;
	BOOL debug, compressed;
	UNITS units;
	PCHAR p;
	PCHAR conf = (PCHAR) NULL;
	WEATHER weather;

	progname = strrchr(argv[0], '/');
	if(progname != (PCHAR) NULL)
		progname++;
	else
		progname = argv[0];
	strlwr(progname);

	/* Process input options */

	compressed = FALSE;
	debug = FALSE;
	delim = '@';
	units = UNITS_NONE;

	opterr = 0;			/* Suppress messages */
	while((ch = getopt(argc, argv, ":c:d:Dhl:u:z")) != -1) {
		switch(ch) {
			case 'c':	/* Config file */
				if(conf != (PCHAR) NULL) {
					error(
						"configuration file"
						" specified "
						"more than once");
					exit(EXIT_FAILURE);
				}
				conf = xmalloc(sizeof(optarg)+1);
				strcpy(conf, optarg);
				break;

			case 'd':	/* Specify delimiter */
				if(strlen(optarg) != 1) {
					error("bad delimiter");
					exit(EXIT_FAILURE);
				}
				delim = optarg[0];
				break;

			case 'D':	/* Debug - display trace */
				debug = TRUE;
				break;

			case 'h':	/* Display help */
				putusage();
				exit(EXIT_SUCCESS);
				break;

			case 'l':	/* Location */
				if(location != -1) {
					error(
						"-l specified "
						"more than once");
					exit(EXIT_FAILURE);
				}
				location = (INT) strtol(optarg, NULL, 10);
				break;

			case 'u':	/* Units */
				if(units != UNITS_NONE) {
					error(
						"units type specified"
						" more than once");
						exit(EXIT_FAILURE);
				}
				if(starts_with("imperial", optarg))
					units = IMPERIAL;
				if(starts_with("metric", optarg))
					units = METRIC;
				if(units == UNITS_NONE) {
					error(
						"invalid value "
						"after -u");
					exit(EXIT_FAILURE);
				}
				break;

			case 'z':	/* compressed output */
				compressed = TRUE;
				break;

			case ':':	/* Missing value */
				error("missing value after -%c", optopt);
				exit(EXIT_FAILURE);

			case '?':	/* Invalid flag letter */
				error("invalid flag -%c", optopt);
				exit(EXIT_FAILURE);

			default:
				error("use -h for help");
				exit(EXIT_FAILURE);
		}
	}

	argc -= optind;			/* Move to non-flag arguments */
	argv += optind;
	if(argc != 0) {
		error(
			"too many args");
			exit(EXIT_FAILURE);
	}

	if(conf == (PCHAR) NULL) conf = DEFAULT_CONFIG;

	/* Read configuration */

	rc = read_config(conf, &config);
	if(rc != 0) {
		error(
			"%d configuration error%s",
			rc, rc == 1 ? "" : "s");
		exit(EXIT_FAILURE);
	}

	/* Fill in overrides */

	if(location != -1) config.location = location;
	if(units != UNITS_NONE) config.units = units;

	/* Check for mandatory settings */

	if(config.location == 0) {
		error(0,
			"there is no LOCATION entry in the "
			"configuration file, and no -l option"
			" is present");
		exit(EXIT_FAILURE);
	}

	/* Fill in the rest */

	config.compressed = compressed;
	config.debug = debug;
	config.delim = delim;

	/* Get the weather */

	p = get_weather(&config, &len);
	if(p == (PCHAR) NULL) {
		error("failed to get weather");
		exit(EXIT_FAILURE);
	}

	/* Optionally dump out the data */

	if(config.debug == TRUE) {
		fputs("Raw XML data:\n", stdout);
		fwrite(p, len, 1, stdout);
		fputc('\n', stdout);
	}
	
	/* Extract the data */

	extract_weather(p, len, &config, &weather);
	free(p);
	if(weather.valid == FALSE) {
		error("error extracting data");
		exit(EXIT_FAILURE);
	}

	/* Massage the units as required */

	massage_weather(&config, &weather);

	/* Output the weather */

	output_weather(&config, &weather);

	/* Tidy up and exit */

	return(EXIT_SUCCESS);
}


/*
 * Print message on standard error in printf style,
 * accompanied by program name.
 *
 */

VOID error(PCHAR mes, ...)
{	va_list ap;

	fprintf(stderr, "%s: ", progname);

	va_start(ap, mes);
	vfprintf(stderr, mes, ap);
	va_end(ap);

	fputc('\n', stderr);
}


/*
 * Output program usage information.
 *
 */

static VOID putusage(VOID)
{	PCHAR *p = (PCHAR *) helpinfo;
	PCHAR q;

	for(;;) {
		q = *p++;
		if(*q == '\0') break;

		fprintf(stderr, q, progname);
		fputc('\n', stderr);
	}
	fprintf(stderr, "\nThis is version %d.%d\n", VERSION, EDIT);
}


/*
 * Log a message.
 *
 */

VOID dolog(PCHAR mes, ...)
{	va_list ap;

	fprintf(stderr, "%s: ", progname);

	va_start(ap, mes);
	vfprintf(stderr, mes, ap);
	va_end(ap);

	fputc('\n', stderr);
}


/*
 * Determine if 'little' is a leading substring of 'big',
 * in a case-independent comparison..
 * The comparison fails if 'little' is longer than 'big'.
 *
 * Inputs:
 *	little	supposedly smaller string
 *	big	larger string, of which 'little' is
 *		checked to be a leading substring
 *
 * Outputs:
 *	TRUE	'big' starts with 'little'
 *	FALSE	'big' does not start with 'little'
 *
 */

static BOOL starts_with(PCHAR big, PCHAR little)
{	PCHAR p;

	if(strlen(little) > strlen(big)) return(FALSE);

	p = strstr(big, little);
	if(p != big) return(FALSE);

	return(TRUE);
}


/*
 * Convert a string to lower case, in situ.
 *
 */

VOID strlwr(PCHAR s)
{	PCHAR p = s;
	INT c;

	while(*p != '\0') {
		c = tolower(*p);
		*p++ = c;
	}
}


/*
 * Allocate memory using 'malloc'; terminate with a message
 * if allocation failed.
 *
 */

PVOID xmalloc(size_t size)
{	PVOID res;

	res = malloc(size);

	if(res == (PVOID) NULL)
		error("cannot allocate memory");

	return(res);
}

/*
 * End of file: owrep.c
 *
 */
