/*
 * File: config.c
 *
 * POP3D daemon for distributing mail on Tavi network; to be invoked only
 * by inetd.
 *
 * Configuration file handler.
 *
 * Bob Eager   May 2016
 *
 */

#include <stdio.h>
#include <stdlib.h>
#include <stdarg.h>
#include <string.h>

#include "pop3d.h"
#include "confcmds.h"

#define	MAXLINE		200		/* Maximum length of a config line */

/* Forward references */

static	VOID	config_error(INT, PCHAR, ...);
static	INT	getcmd(PCHAR);


/*
 * Read and parse the configuration file specified by 'configfile'.
 *
 * Returns:
 *	Number of errors encountered.
 *	Any error messages have already been issued.
 *
 * The configuration information is returned in the structure 'config'.
 *
 */

INT read_config(PCHAR configfile, PCONFIG config)
{	INT line = 0;
	INT errors = 0;
	PCHAR p, q, r, s, temp;
	FILE *fp;
	CHAR buf[MAXLINE];

	/* Set defaults */

	memset(config, 0, sizeof(CONFIG));
	config->log_type = LOGGING_FILE;
	config->log_file = (PCHAR) NULL;
	config->split_subject = FALSE;

	fp = fopen(configfile, "r");
	if(fp == (FILE *) NULL) {
		config_error(0, "cannot open configuration file %s", configfile);
		return(++errors);
	}

	for(;;) {
		p = fgets(buf, MAXLINE, fp);
		if(p == (PCHAR) NULL) break;
		temp = p + strlen(p) - 1;	/* Point to last character */
		if(*temp == '\n') *temp = '\0';	/* Remove any newline */
		line++;

		p = strchr(buf, '#');		/* Strip comments */
		if(p != (PCHAR) NULL) *p = '\0';

		p = strtok(buf, " \t");
		q = strtok(NULL, " \t");
		r = strtok(NULL, " \t");
		s = strtok(NULL, " \t");

		/* Skip non-information lines */

		if((p == (PCHAR) NULL) ||	/* No tokens */
		   (*p == '\n'))		/* Empty line */
			continue;

		switch(getcmd(p)) {
			case CMD_LOGGING:
				if(r != (PCHAR) NULL) {
					config_error(
						line,
						"syntax error (extra on end)");
					errors++;
					continue;
				}
				strlwr(q);
				if(strcmp(q, "file") == 0) {
					config->log_type = LOGGING_FILE;
					continue;
				}
				if(strcmp(q, "syslog") == 0) {
					config->log_type = LOGGING_SYSLOG;
					continue;
				}
				config_error(
					line,
					"unrecognised logging type '%s'",
					q);
				errors++;
				break;

			case CMD_LOGFILE:
				if(r != (PCHAR) NULL) {
					config_error(
						line,
						"syntax error (extra on end)");
					errors++;
					continue;
				}
				if(config->log_file != (PCHAR) NULL) {
					config_error(
						line,
						"logfile name specified "
						"more than once");
					errors++;
					continue;
				}
				config->log_file = strdup(q);
				break;

			case CMD_MAILBASE:
				if(r != (PCHAR) NULL) {
					config_error(
						line,
						"syntax error (extra on end)");
					errors++;
					continue;
				}
				if(config->mail_base != (PCHAR) NULL) {
					config_error(
						line,
						"mail base directory specified "
						"more than once");
					errors++;
					continue;
				}
				config->mail_base = strdup(q);
				break;

			case CMD_SPLITSUBJ:
				if(r != (PCHAR) NULL) {
					config_error(
						line,
						"syntax error (extra on end)");
					errors++;
					continue;
				}
				strlwr(q);
				if(strcmp(q, "yes") == 0) {
					config->split_subject = TRUE;
					continue;
				}
				if(strcmp(q, "no") == 0) {
					config->split_subject = FALSE;
					continue;
				}
				config_error(
					line,
					"unrecognised value for SPLIT_SUBJECT"
					" - '%s'",
					q);
				break;

			case CMD_USER:
				if(r != (PCHAR) NULL) {
					config_error(
						line,
						"syntax error (extra on end)");
					errors++;
					continue;
				}
				if(config->user != (PCHAR) NULL) {
					config_error(
						line,
						"user specified "
						"more than once");
					errors++;
					continue;
				}
				config->user = strdup(q);
				break;

			case CMD_GROUP:
				if(r != (PCHAR) NULL) {
					config_error(
						line,
						"syntax error (extra on end)");
					errors++;
					continue;
				}
				if(config->group != (PCHAR) NULL) {
					config_error(
						line,
						"group specified "
						"more than once");
					errors++;
					continue;
				}
				config->group = strdup(q);
				break;

			default:
				config_error(
					line,
					"unrecognised command '%s'", p);
				errors++;
				break;
		}
	}

	fclose (fp);

	if(config->log_type == LOGGING_FILE &&
	   config->log_file == (PCHAR) NULL)
		config->log_file = DEFAULT_LOGFILE;
	if(config->mail_base == (PCHAR) NULL)
		config->mail_base = DEFAULT_MAILBASE;
#ifdef	DEBUG
	syslog(LOG_DEBUG, "config: logging = '%s'\n",
		config->log_type == LOGGING_FILE ? "file" : "syslog");
	syslog(LOG_DEBUG, "config: logfile = |%s|\n", config->log_file);
	syslog(LOG_DEBUG, "config: mailbase = |%s|\n", config->mail_base);
	syslog(LOG_DEBUG, "config: split_subject = '%s'\n",
		config->split_subject == TRUE ? "yes" : "no");
	syslog(LOG_DEBUG, "config: user = |%s|\n",
		config->user == (PCHAR) NULL ? (PCHAR) "(unset)" : config->user);
	syslog(LOG_DEBUG, "config: group = |%s|\n",
		config->group == (PCHAR) NULL ? (PCHAR) "(unset)" : config->group);
#endif

	return(errors);
}


/*
 * Check command in 's' for validity, and return command code.
 * Case is immaterial.
 *
 * Returns CMD_BAD if command not recognised.
 *
 */

static INT getcmd(PCHAR s)
{	INT i;

	for(i = 0; ; i++) {
		if(cmdtab[i].cmdcode == CMD_BAD) return(CMD_BAD);
		if(strcasecmp(s, cmdtab[i].cmdname) == 0) break;
	}

	return(cmdtab[i].cmdcode);
}


/*
 * Output configuration error message to standard error in printf style.
 *
 */

static VOID config_error(INT line, PCHAR mes, ...)
{	va_list ap;
	CHAR buf[MAXLOG];

	va_start(ap, mes);
	vsprintf(buf, mes, ap);
	va_end(ap);

	if(line == 0)
		error("config: %s", buf);
	else
		error("config: line %d: %s", line, buf);
}

/*
 * End of file: config.c
 *
 */
