/*
 * File: maildrop.c
 *
 * General maildrop handling routines.
 *
 * Bob Eager   May 2016
 *
 */

#include <dirent.h>
#include <sys/stat.h>
#include <stdlib.h>
#include <stdio.h>
#include <string.h>
#include <ctype.h>
#include <unistd.h>

#include "pop3d.h"
#include "maildrop.h"
#include "lock.h"
#include "netio.h"

#define	SLOTQ		50		/* Message slot quantum */

#define	MAXMES		100		/* Maximum length of a return message */

#define	MAXLOCKTRIES	5		/* Maximum number of tries at maildrop lock */
#define	LOCKINTERVAL	10		/* Lock retry interval (seconds) */

#define	NEWDIR		"/new"

#define	STR_SUBJECT	"Subject:"
#define	STR_XSPLIT	"X-Split-Subject: "
#define	SUBJ_SPLIT	78		/* Line length at which to (optionally)
					   split Subject: lines */

/* Type definitions */

typedef struct _MSG {			/* Message table entry */
ULONG		size;
INT		delete;
PCHAR		name;
} MSG, *PMSG;

typedef	struct _LF {			/* Line fragment */
struct _LF	*link;			/* Link to next fragment */
CHAR		data[SUBJ_SPLIT+2];	/* Line data (+ newline and null) */
} LF, *PLF;

/* Forward references */

static	BOOL	build_table(VOID);
static	INT	sortfunc(const void *, const void *);

/* Local storage */

static	PCHAR	maildir = (PCHAR) NULL;
static	INT	msgcount;
static	ULONG	total_octets;
static	PCHAR	userdir = (PCHAR) NULL;
static	PMSG	msgtable;
static	FILE	*mailfp;
static	PLF	lq;			/* Line queue */
static	BOOL	in_header;


/*
 * Initialise storage, etc.
 *
 * Returns:
 *	MAILINIT_OK		Initialisation successful
 *	MAILINIT_NOENV		Mail spool directory env variable not set
 *	MAILINIT_BADDIR		Cannot access mail spool directory
 *
 */

INT mail_init(PCHAR mailbase)
{	INT last;
	INT rc;

	maildir = strdup(mailbase);
	last = strlen(maildir) - 1;
	if(maildir[last] == '/')
		maildir[last] = '\0';
	rc = chdir(maildir);
	if(rc != 0) return(MAILINIT_BADDIR);
#ifdef	DEBUG
	trace("mail base directory = \"%s\"\n", maildir);
#endif

	mailfp = (FILE *) NULL;
	msgtable = (PMSG) NULL;

	return(MAILINIT_OK);
}


/*
 * Set up for a new username.
 *
 * Returns:
 *	MAILUSER_OK		new user selected and maildrop locked
 *	MAILUSER_BADDIR		no user maildrop directory
 *	MAILUSER_NOLOCK		could not lock maildrop
 *	MAILUSER_FAIL		other failure
 *
 */

INT mail_user(PCHAR username)
{	INT rc;
	INT i;

	/* Select the maildrop directory */

	userdir = xmalloc(strlen(maildir) +
		  1 +			/* "/" */
		  strlen(username) +
		  strlen(NEWDIR) +
		  1);			/* '\0' */
	strcpy(userdir, maildir);
	strcat(userdir, "/");
	strcat(userdir, username);
	strcat(userdir, NEWDIR);

#ifdef	DEBUG
	trace("mail user directory = \"%s\"", userdir);
#endif

	rc = chdir(userdir);
	if(rc != 0) return(MAILUSER_BADDIR);

	/* Lock the maildrop directory */

	for(i = 1; i <= MAXLOCKTRIES; i++) {
#ifdef	DEBUG
		trace("trying for maildrop lock...");
#endif
		rc = lock(userdir);
		if(rc == TRUE) break;
		if(i == MAXLOCKTRIES) return(MAILUSER_NOLOCK);

#ifdef	DEBUG
		trace("waiting for lock...");
#endif
		sleep(LOCKINTERVAL);
	}

#ifdef	DEBUG
	trace("locked the maildrop");
#endif
	/* Build the message table */

	if(build_table() == FALSE) return(MAILUSER_FAIL);

	return(MAILUSER_OK);
}


/*
 * Build the table of messages in the maildrop.
 *
 */

static BOOL build_table(VOID)
{	DIR *dirp;
	INT freeslots = 0;
	struct dirent *entry;
	struct stat st;
	INT rc;
	INT len;
	PMSG cur;
#ifdef	DEBUG
	CHAR db[30];
#endif

	msgcount = 0;
	total_octets = 0;
	dirp = opendir(".");
	if(dirp == (DIR *) NULL) {
		error("open spool directory failed");
		return(FALSE);
	}

#ifdef	DEBUG
	trace("starting directory scan...");
#endif
	for(;;) {
		entry = readdir(dirp);
		if(entry == (struct dirent *) NULL) break;
		if(entry->d_type != DT_REG) continue;
#ifdef	DEBUG
		trace("msgcount=%d, freeslots=%d", msgcount, freeslots);
#endif
		if(freeslots <= 0) {
#ifdef	DEBUG
			trace("creating slots");
#endif
			msgtable = realloc(msgtable,
				(msgcount+SLOTQ)*sizeof(MSG));
			if(msgtable == (PMSG) NULL) {
				error("out of storage");
				return(FALSE);
			}
			freeslots = SLOTQ;
		}

		cur = &msgtable[msgcount];
		len = strlen(entry->d_name);
		cur->name = (PCHAR) xmalloc(len+1);
		if(cur->name == (PCHAR) NULL) {
			error("out of storage");
			return(FALSE);
		}
		strcpy(cur->name, entry->d_name);
		rc = stat(entry->d_name, &st);
		if(rc != 0) st.st_size = 9999;
		cur->size = st.st_size;
		cur->delete = FALSE;
		msgcount++;
		freeslots--;

	}
	closedir(dirp);

#ifdef	DEBUG
	{	INT i;

		trace("     Before sort");
		trace("  msg   size   name");
		for(i = 0; i < msgcount; i++) {
			strncpy(db, msgtable[i].name, 27);
			db[27] = '.';
			db[28] = '.';
			db[29] = '\0';
			trace("%4d %6d  %s", i+1, msgtable[i].size, db);
		}
		trace("------------------------------------------");
	}
#endif

	/* Sort the files into alphabetical order,
	   which corresponds to likely arrival order. */

	qsort(&msgtable[0], msgcount, sizeof(MSG), sortfunc);

#ifdef	DEBUG
	{	INT i;

		trace("     After sort");
		trace("  msg   size   name");
		for(i = 0; i < msgcount; i++) {
			strncpy(db, msgtable[i].name, 27);
			db[27] = '.';
			db[28] = '.';
			db[29] = '\0';
			trace("%4d %6d  %s", i+1, msgtable[i].size, db);
		}
		trace("--------------------------");
	}
#endif

	return(TRUE);
}


/*
 * Function used for sorting the message table.
 * This is used by 'qsort', and must return:
 *	< 0	p less than q
 *	= 0	p equal to q
 *	> 0	p greater than q
 *
 */

static INT sortfunc(const void *p, const void *q)
{	return(strcasecmp(((PMSG)p)->name, ((PMSG)q)->name));
}


/*
 * Return maildrop statistics.
 *
 * These consist of the number of messages, and the total number of octets.
 *
 */

VOID mail_stat(PINT nmsgs, PINT octets)
{	INT i;

	*nmsgs = msgcount;
	if(total_octets == 0) {
		for(i = 0; i < msgcount; i++)
			total_octets += msgtable[i].size;
	}
	*octets = total_octets;
#ifdef	DEBUG
	trace("stats: %d messages, %d octets", msgcount, total_octets);
#endif
}


/*
 * Return mail item statistics.
 *
 * This consists of the size of a message in octets, or -1 if the message
 * does not exist. Note that the message number is 1-based.
 * The unique ID for the message is also returned, by copying via the
 * ID pointer if it is non-null.
 *
 */

VOID mail_info(INT msg, PINT octets, PCHAR id)
{	PMSG item = &msgtable[msg-1];
	PCHAR p;
	INT i;

	if(item->delete == TRUE) {
		*octets = -1;
	} else {
		*octets = item->size;
		if(id != (PCHAR) NULL) {	/* Copy ID as well */
			for(p = item->name, i = 0; ; p++) {
				*id++ = *p;
				if(*p == '\0') break;
				if(++i == MAXID) {
					*id = '\0';
					break;
				}
			}
		}
	}
}


/*
 * Commence retrieval of a mail item. This opens the mail file ready for
 * access. Note that the message number is 1-based.
 *
 * Returns:
 *	MAILRETR_OK	mail ready for retrieval
 *	MAILRETR_FAIL	failed to access mail file
 *	MAILRETR_NOMES	no such message
 *
 */

INT mail_retr(INT msg, PINT octets)
{	PMSG item;
	CHAR mes[MAXMES+1];

	if(msg <= 0 || msg > msgcount) return(MAILRETR_NOMES);

	item = &msgtable[msg-1];

	/* Get basic statistics */

	if(item->delete == TRUE) {
		return(MAILRETR_NOMES);
	} else {
		*octets = item->size;
	}

	/* Open the mail file */

	mailfp = fopen(item->name, "r");
	if(mailfp == (FILE *) NULL) {
		sprintf(mes, "cannot open mail file %s", item->name);
		error(mes);
		dolog(LOG_ERR, mes);
		return(MAILRETR_FAIL);
	}

	lq = (PLF) NULL;		/* Initialise subject fragment queue */
	in_header = TRUE;		/* Now in message header */

	return(MAILRETR_OK);
}


/*
 * Retrieve the next line from the current mail file.
 * Performs dot stuffing as required, and generates the final dot line.
 *
 * Returns:
 *	MAILLINE_OK	Line retrieved OK
 *	MAILLINE_EOF	End of file, returning dot line
 *	MAILLINE_FAIL	Error retrieving mail line
 *
 * If the line is a Subject: line, and split_subject is TRUE, split the
 * line after SUBJ_SPLIT characters. The remainder of that line is returned
 * on subsequent calls, converted to X-Split-Subject:.
 *
 */

INT mail_line(PCHAR buf, INT max, BOOL split_subject)
{	INT len, n;
	PCHAR p, q;
	PLF tq;
	PLF pq = (PLF) NULL;

	if(lq != (PLF) NULL) {		/* Line(s) queued */
		if(strlen(lq->data) + 1 > max)
			return(MAILLINE_FAIL);	/* Should never happen */
		strcpy(buf, lq->data);	/* Copy in the actual line */
#ifdef	DEBUG
		trace("subjfrag_out: %s", lq->data);
#endif
		tq = lq;		/* Save for free */
		lq = lq->link;		/* Unlink this item */
		free((PVOID) tq);	/* Return queued item */
		return(MAILLINE_OK);
	}

	p = fgets(buf, max, mailfp);
	if(p == (PCHAR) NULL) {		/* End of file */
		buf[0] = '.';
		buf[1] = '\n';
		buf[2] = '\0';

		return(MAILLINE_EOF);
	}

	len = strlen(buf);
	if((in_header == TRUE) && (buf[0] == '\n')) {
#ifdef	DEBUG
		trace("end of header found");
#endif
		in_header = FALSE;	/* Reached message body */
	}

	/* Split Subject: lines if required */

	if((split_subject == TRUE) &&
		(in_header == TRUE ) &&
		(len > SUBJ_SPLIT+1) &&
		(strncasecmp(STR_SUBJECT, buf, strlen(STR_SUBJECT)) == 0)) {
#ifdef	DEBUG
		trace("splitting subject, length=%d, |%s|", len, buf);
#endif
		lq = (PLF) NULL;		/* Initialise queue */
		if(buf[len-1] == '\n')
			buf[len-1] = '\0';	/* Remove newline */
		p = &buf[SUBJ_SPLIT];
		q = p;		/* Save point of first split */
		while(strlen(p) > 0) {
#ifdef	DEBUG
			trace("splitremain: %d", strlen(p));
#endif
			len = SUBJ_SPLIT + 1;	/* Need a newline too */
			n = SUBJ_SPLIT - strlen(STR_XSPLIT);
						/* Size of chunk to take */
			if(n > strlen(p)) n = strlen(p);
#ifdef	DEBUG
			trace("subjfrag: size=%d", n);
#endif
			tq = (PLF) malloc(sizeof(LF));
			if(tq == (PLF) NULL) return(MAILLINE_FAIL);
			strcpy(tq->data, STR_XSPLIT);
			strncat(tq->data, p, n);
			strcat(tq->data, "\n");
#ifdef	DEBUG
			trace("frag: %s", tq->data);
#endif
			tq->link = (PLF) NULL;
			if(lq == (PLF) NULL) {
				lq = tq;
			} else {
				pq->link = tq;
			}
			pq = tq;
			p += n;			/* Move to next chunk */
		}
		*q++ = '\n';
		*q = '\0';	/* Truncate first part here */
		len = strlen(buf);
	}

	if((buf[len-1] != '\n') ||
	   ((buf[0] == '.') && (len == max))) {
		strcpy(buf, "line too long in mail file");
#if 0
			error(buf);
#endif
			dolog(LOG_ERR, buf);
		return(MAILLINE_FAIL);
	}

	if(buf[0] == '.') {		/* Need to dot stuff */
		memmove(&buf[1], &buf[0], len+1);
		buf[0] = '.';
	}

#ifdef	DEBUG
	trace("mail_line: %s", buf);
#endif
	return(MAILLINE_OK);
}


/*
 * Close any open mail file.
 *
 */

VOID mail_close(VOID)
{	if(mailfp != (FILE *) NULL) {
		(VOID) fclose(mailfp);
		mailfp = (FILE *) NULL;
	}
}


/*
 * Perform a reset. Clear all delete indicators.
 *
 */

VOID mail_rset(VOID)
{	INT i;

	for(i = 0; i < msgcount; i++)
		msgtable[i].delete = FALSE;
}


/*
 * Delete a message. Strictly, the message is just marked for deletion.
 *
 * Returns:
 *	MAILDELE_OK	message marked for deletion
 *	MAILDELE_FAIL	no such message
 *
 */

INT mail_dele(INT msgno)
{	if(msgno < 0 ||
	   msgno > msgcount ||
	   msgtable[msgno-1].delete == TRUE)
		return(MAILDELE_FAIL);

	msgtable[msgno-1].delete = TRUE;
	return(MAILDELE_OK);
}


/*
 * Close down the maildrop; commit any deletions,
 * and unlock the maildrop if required.
 *
 */

VOID mail_commit(INT unlocking, PINT count, PINT octets)
{	INT i;
	PMSG cur;

	mail_close();			/* For safety */
	*count = 0;
	*octets = 0;

#ifdef	DEBUG
	trace("commit: msgcount = %d", msgcount);
#endif
	for(i = 0; i < msgcount; i++) {
		cur = &msgtable[i];
		if(cur->delete == FALSE) {
			*count += 1;
			*octets += cur->size;
			continue;
		}

#ifdef	DEBUG
		trace("committing deletion of message %d, file %s",
			i+1, cur->name);
#endif
		remove(cur->name);
	}

	if(unlocking == TRUE) unlock();

	if(maildir != (PCHAR) NULL)
		free(maildir);
	if(userdir != (PCHAR) NULL)
		free(userdir);
}

/*
 * End of file: maildrop.c
 *
 */
