/*
 * File: pop3d.c
 *
 * POP3D daemon for distributing mail on Tavi network; normally invoked
 * by inetd.
 *
 * Main program
 *
 * Bob Eager   May 2016
 *
 */

/*
 * History:
 *	5.0	Initial version for FreeBSD.
 *	5.1	Fix for unsigned char compilation warnings.
 *	5.2	Added PID to syslog entries.
 *	5.3	Suppressed error message on over-long lines.
 *	5.4	Fixed select problem (wrong nfds) for FreeBSD 8.3.
 *	5.5	Minor fixes for clang compiler.
 *
 */

#include <ctype.h>
#include <errno.h>
#include <stdarg.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <time.h>
#include <unistd.h>
#include <pwd.h>
#include <grp.h>
#include <sys/types.h>
#include <sys/socket.h>
#include <netdb.h>
#include <netinet/in.h>
#include <arpa/inet.h>
#include <arpa/nameser.h>
#include <resolv.h>

#include "pop3d.h"

#define	STDIN	0

/* Forward references */

static	VOID	fix_domain(PCHAR);
static	VOID	log_connection(VOID);

/* Local storage */

static	CONFIG	config;
static	CHAR 	hostname[MAXHOSTNAMELEN+1];
static	CHAR 	myname[MAXHOSTNAMELEN+1];
static	PCHAR	progname;


/*
 * Parse arguments and handle options.
 *
 */

INT main(INT argc, CHAR **argv)
{	socklen_t namelen;
	INT rc, i;
	uid_t uid;
	gid_t gid;
	SOCK client;
	PHOST host;
	PCHAR argp, p;
	PCHAR conf = (PCHAR) NULL;
	struct passwd *pw;
	struct group *gr;

	progname = strrchr(argv[0], '/');
	if(progname != (PCHAR) NULL)
		progname++;
	else
		progname = argv[0];
	p = strchr(progname, '.');
	if(p != (PCHAR) NULL) *p = '\0';
	strlwr(progname);

	tzset();			/* Set time zone */
	res_init();			/* Initialise resolver */

	/* Process input options */

	for(i = 1; i < argc; i++) {
		argp = argv[i];
		if(argp[0] == '-') {		/* Option */
			switch(argp[1]) {
				case 'c':	/* Config file */
					if(conf != (PCHAR) NULL) {
						error(
							"configuration file"
							" specified "
							"more than once");
						exit(EXIT_FAILURE);
					}
					if(argp[2] != '\0') {
						conf = (PCHAR) strdup(&argp[2]);
					} else {
						if(i == argc - 1) {
							error(
								"no arg for -c");
							exit(EXIT_FAILURE);
						} else {
								conf = (PCHAR)
								strdup(argv[++i]);
						}
					}
					break;

				case '\0':
					error("missing option after '-'");
					exit(EXIT_FAILURE);

				default:
					error("invalid option '-%c'", argp[1]);
					exit(EXIT_FAILURE);
			}
		} else {
			error("invalid argument '%s'", argp);
			exit(EXIT_FAILURE);
		}
	}

	if(conf == (PCHAR) NULL) conf = DEFAULT_CONFIG;
#ifdef	DEBUG
	{	CHAR s[100];
		s[0] = '\0';
		for(i = 1; i < argc; i++) {
			strcat(s, argv[i]); strcat(s, " ");
		}
		syslog(LOG_DEBUG, "========================"); 
		syslog(LOG_DEBUG, "   |  pop3d  |  pop3d  |");
		syslog(LOG_DEBUG, "   |         |         |");
		syslog(LOG_DEBUG, "   V         V         V");
		syslog(LOG_DEBUG, "args = |%s|", s);
		syslog(LOG_DEBUG, "config file = |%s|", conf);
	}
#endif

	/* Get IP address of the client */

	client.sin_family = AF_INET;
	namelen = sizeof(SOCK);
	rc = getpeername(STDIN, (PSOCKG) &client, &namelen);
	if(rc != 0) {
		error("cannot get peer name, errno = %d", errno);
		exit(EXIT_FAILURE);
	}

	/* Get the host name of this server; if not possible, set it to the
	   dotted address. */

	rc = gethostname(myname, sizeof(myname));
	if(rc != 0) {
		INADDR myaddr;

		myaddr.s_addr = htonl(gethostid());
		sprintf(myname, "[%s]", (PCHAR) inet_ntoa(myaddr));
	} else {
		fix_domain(myname);
	}

	/* Get the host name of the client; if not possible, set it to the
	   dotted address */

	host = gethostbyaddr((PCHAR) &client.sin_addr,
			     sizeof(client.sin_addr), AF_INET);
	if(host == (PHOST) NULL) {
		if(h_errno == HOST_NOT_FOUND) {
			sprintf(hostname, "[%s]",
				(PCHAR) inet_ntoa(client.sin_addr));
		} else {
			error("cannot get host name, errno = %d", h_errno);
			exit(EXIT_FAILURE);
		}
	} else {
		strcpy(hostname, host->h_name);
		fix_domain(hostname);
	}

	/* Read configuration */

	rc = read_config(conf, &config);
	if(rc != 0) {
		error("%d configuration error%s", rc, rc == 1 ? "" : "s");
		exit(EXIT_FAILURE);
	}

	/* Set up uid and gid */

	uid = geteuid();
	gid = getegid();
#ifdef	DEBUG
	syslog(LOG_DEBUG, "initial uid=%d, gid=%d", uid, gid);
#endif

	if(config.user != (PCHAR) NULL) {
		pw = getpwnam(config.user);
		if(pw == (struct passwd *) NULL) {
			error("cannot run as %s; no such user", config.user);
			exit(EXIT_FAILURE);
		}
		uid = pw->pw_uid;
	}
	if(config.group != (PCHAR) NULL) {
		gr = getgrnam(config.group);
		if(gr == (struct group *) NULL) {
			error("cannot run in group %s; no such group",
				config.group);
			exit(EXIT_FAILURE);
		}
		gid = gr->gr_gid;
	}
#ifdef	DEBUG
	syslog(LOG_DEBUG, "final uid=%d, gid=%d", uid, gid);
#endif

	rc = setgid(gid);
	if(rc != 0) {
		error("cannot set gid to %s", config.group);
		exit(EXIT_FAILURE);
	}
	rc = setuid(uid);
	if(rc != 0) {
		error("cannot set uid to %s", config.user);
		exit(EXIT_FAILURE);
	}

	/* Start logging */

	rc = open_log(config.log_type, config.log_file, myname, progname);
	if(rc != LOGERR_OK) {
		error(
		"logging initialisation failed - %s",
		rc == LOGERR_OPENFAIL ? "file open failed" :
					"internal log type failure");
		exit(EXIT_FAILURE);
	}

	log_connection();

	/* Run the server */

	rc = server(STDIN, hostname, myname, &config);

	/* Shut down */

	(VOID) close(STDIN);
	close_log();

	return(rc == 0 ? EXIT_SUCCESS : EXIT_FAILURE);
}


/*
 * Send an error message to syslog.
 *
 */

VOID error(PCHAR mes, ...)
{	va_list ap;
	CHAR s[200];

	va_start(ap, mes);
	vsprintf(s, mes, ap);
	va_end(ap);

	syslog(LOG_CONSOLE | LOG_ERR, "%s", s);
}


/*
 * Check for a full domain name; if not present, add default domain name.
 *
 */

static VOID fix_domain(PCHAR name)
{	if(strchr(name, '.') == (PCHAR) NULL && _res.defdname[0] != '\0') {
		strcat(name, ".");
		strcat(name, _res.defdname);
	}
}


/*
 * Log details of the connection.
 *
 */

static VOID log_connection(VOID)
{	time_t tod;
	CHAR timeinfo[35];
	CHAR buf[100];

	(VOID) time(&tod);
	(VOID) strftime(timeinfo, sizeof(timeinfo),
		"on %a %d %b %Y at %X %Z", localtime(&tod));

	sprintf(buf, "connection from %s", hostname);

	dolog(LOG_INFO, buf);
}


/*
 * Allocate memory using 'malloc'; terminate with a message
 * if allocation failed.
 *
 */

PVOID xmalloc(size_t size)
{	PVOID res;

	res = malloc(size);

	if(res == (PVOID) NULL)
		error("cannot allocate memory");

	return(res);
}


/*
 * Convert a string to lower case, in situ.
 *
 */

VOID strlwr(PCHAR s)
{	PCHAR p = s;
	INT c;

	while(*p != '\0') {
		c = tolower(*p);
		*p++ = c;
	}
}

/*
 * End of file: pop3d.c
 *
 */
